package com.wecloud.im.biz.module.message.controller;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

import java.util.List;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.wecloud.can.core.base.controller.BaseController;
import com.wecloud.can.core.enums.ApiCode;
import com.wecloud.can.core.util.ApiResult;
import com.wecloud.can.datasource.mybatisplus.pagination.Paging;
import com.wecloud.im.biz.module.message.controller.param.GetReadersParam;
import com.wecloud.im.biz.module.message.controller.param.GroupChatMessageParam;
import com.wecloud.im.biz.module.message.controller.param.GroupChatStatusMessageParam;
import com.wecloud.im.biz.module.message.controller.param.ImHistoryMessagePageParam;
import com.wecloud.im.biz.module.message.controller.param.MsgDeleteParam;
import com.wecloud.im.biz.module.message.controller.param.PrivateChatMessageParam;
import com.wecloud.im.biz.module.message.controller.param.PrivateChatStatusMessageParam;
import com.wecloud.im.biz.module.message.controller.param.SingleUserNotificationParam;
import com.wecloud.im.biz.module.message.controller.param.add.ImMsgRecall;
import com.wecloud.im.biz.module.message.controller.param.add.ImMsgUpdate;
import com.wecloud.im.biz.module.message.controller.vo.ImMessageOfflineListVo;
import com.wecloud.im.biz.module.message.controller.vo.OfflineMsgDto;
import com.wecloud.im.biz.module.message.controller.vo.ReaderList;
import com.wecloud.im.biz.module.message.service.ImMessageService;

/**
 * 消息存储表 控制器
 *
 * @author wei
 * @since 2021-04-29
 */
@Slf4j
@RestController
@RequestMapping("/imMessage")
@Api(value = "消息存储表API", tags = {"消息存储表"})
public class ImMessageController extends BaseController {

    @Autowired
    private ImMessageService imMessageService;

    /**
     * 消息撤回
     */
    @PostMapping("/withdraw")
    @ApiOperation(value = "消息撤回", notes = "只能撤回客户端自己发送的消息")
    public ApiResult<Boolean> withdrawMsg(@RequestBody ImMsgRecall imMsgRecall) throws Exception {
        return ApiResult.ok(imMessageService.withdrawMsg(imMsgRecall));
    }

    /**
     * 删除消息
     */
    @PostMapping("/delete")
    @ApiOperation(value = "删除消息", notes = "单条消息与批量消息删除共用")
    public ApiResult<Boolean> deleteMsg(@RequestBody MsgDeleteParam param) {
        if (param == null) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
        }
        if (CollectionUtils.isEmpty(param.getMsgIds())) {
            return ApiResult.fail("请选择要删除的消息id");
        }
        return ApiResult.ok(imMessageService.deleteMsg(param));
    }

    /**
     * 修改消息体
     */
    @PostMapping("/updateMsgById")
    @ApiOperation(value = "修改消息体", notes = "只能修改客户端自己发送的消息")
    public ApiResult<Boolean> updateMsgById(@RequestBody ImMsgUpdate imMsgUpdate) throws Exception {
        return ApiResult.ok(imMessageService.updateMsgById(imMsgUpdate));
    }

    /**
     * 离线消息列表
     */
    @PostMapping("/offlineList")
    @ApiOperation(value = "离线消息列表")
    public ApiResult<List<ImMessageOfflineListVo>> getOfflineList() throws Exception {
        List<ImMessageOfflineListVo> offlineList = imMessageService.getOfflineList();
        return ApiResult.ok(offlineList);
    }

    /**
     * 查询某个会话历史消息分页列表
     */
    @PostMapping("/getHistoryMsg")
    @ApiOperation(value = "查询某个会话历史消息分页列表")
    public ApiResult<Paging<OfflineMsgDto>> getHistoryMsg(@Validated @RequestBody ImHistoryMessagePageParam imHistoryMessagePageParam) {
        Paging<OfflineMsgDto> paging = imMessageService.getHistoryMsgConversationId(imHistoryMessagePageParam);
        return ApiResult.ok(paging);
    }

    /**
     * 查询某个消息已读client列表和未读client
     */
    @PostMapping("/getReaders")
    @ApiOperation(value = "查询某个消息已读client列表和未读client")
    public ApiResult<ReaderList> getReaders(@Validated @RequestBody GetReadersParam param) {
        ReaderList readerList = imMessageService.getReaders(param);
        return ApiResult.ok(readerList);
    }

    @ApiOperation("发布群聊消息")
    @PostMapping("/group/publish")
    public ApiResult<Boolean> groupMessagePublish(@RequestBody @Validated GroupChatMessageParam param) {
        // 参数校验
        return ApiResult.ok(imMessageService.groupMessagePublish(param));
    }

    @ApiOperation("发布群聊状态消息")
    @PostMapping("/group/status/publish")
    public ApiResult<Boolean> groupStatusMessagePublish(@RequestBody @Validated GroupChatStatusMessageParam param) {
        return ApiResult.ok(imMessageService.groupStatusMessagePublish(param));
    }

    @ApiOperation("发布单聊消息")
    @PostMapping("/private/publish")
    public ApiResult<Boolean> privateMessagePublish(@RequestBody @Validated PrivateChatMessageParam param) {
        return ApiResult.ok(imMessageService.privateMessagePublish(param));
    }

    @ApiOperation("发布单聊状态消息")
    @PostMapping("/private/status/publish")
    public ApiResult<Boolean> privateStatusMessagePublish(@RequestBody @Validated PrivateChatStatusMessageParam param) {
        return ApiResult.ok(imMessageService.privateStatusMessagePublish(param));
    }

    @ApiOperation("单个用户通知-普通消息")
    @PostMapping("/system-notification/publish")
    public ApiResult<Boolean> singleUserNotification(@RequestBody @Validated SingleUserNotificationParam param) {
        return ApiResult.ok(imMessageService.singleUserNotification(param));
    }

}

