package com.wecloud.im.chatroom.action;

import com.wecloud.dispatch.annotation.ActionMapping;
import com.wecloud.dispatch.common.BaseRequest;
import com.wecloud.dispatch.extend.ActionRequest;
import com.wecloud.im.entity.ImApplication;
import com.wecloud.im.entity.ImClient;
import com.wecloud.im.entity.ImMessageOnlineSend;
import com.wecloud.im.param.ChatContentVo;
import com.wecloud.im.param.ImConversationQueryVo;
import com.wecloud.im.param.MsgVo;
import com.wecloud.im.service.ImApplicationService;
import com.wecloud.im.service.ImClientService;
import com.wecloud.im.service.ImConversationMembersService;
import com.wecloud.im.service.ImConversationService;
import com.wecloud.im.thousandchat.cache.ThousandChatCacheManager;
import com.wecloud.im.ws.enums.WsResponseCmdEnum;
import com.wecloud.im.ws.model.WsResponse;
import com.wecloud.im.ws.sender.ChannelSender;
import com.wecloud.utils.JsonUtils;
import com.wecloud.utils.SnowflakeUtil;
import io.geekidea.springbootplus.framework.common.api.ApiCode;
import io.geekidea.springbootplus.framework.common.api.ApiResult;
import io.netty.channel.Channel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * @Author wenzhida
 * @Date 2022/4/26 21:05
 * @Description 聊天室消息处理
 */
@Slf4j
@Component
@ActionMapping
@Api(value = "聊天室消息处理", tags = {"聊天室消息处理"})
public class ChatRoomAction {

    @Autowired
    private ImApplicationService imApplicationService;
    @Autowired
    private ImClientService imClientService;
    @Autowired
    private ImConversationService imConversationService;
    @Autowired
    private ChannelSender channelSender;
    @Autowired
    private ThousandChatCacheManager thousandChatCacheManager;

    @ActionMapping("/chat/chatroom/send")
    @ApiOperation(value = "聊天室消息发送")
    public void sendMsg(ActionRequest request, ChatContentVo data, String reqId) {
        if (log.isDebugEnabled()) {
            log.debug("接收到参数，reqId: {},\n data: {}, ", data);
        }
        ImConversationQueryVo conversation = imConversationService.getCacheImConversationById(data.getToConversation());
        if (conversation == null) {
            log.warn("reqId: {} ,会话id: {} db 中不存在", reqId, data.getToConversation());
            return;
        }
        // 查询发送者client
        ImClient imClientSender = imClientService.getCacheImClient(request.getSenderClientId());
        if (imClientSender == null) {
            log.warn("根据senderClientId: {} 查找不到 imClientSender！", request.getSenderClientId());
            return;
        }
        // 查询imApplication
        ImApplication imApplication = imApplicationService.getCacheById(imClientSender.getFkAppid());
        if (imApplication == null) {
            log.warn("根据appId: {} 查找不到 imApplication！", imClientSender.getFkAppid());
            return;
        }
        ImMessageOnlineSend imMessageOnlineSend = assembleImMessageOnlineSend(data, imClientSender, imApplication.getId());
        // 在线用户直接发消息
        sendMsgForOnline(data, imMessageOnlineSend);
        // 响应发送方消息id等信息
        response(reqId, imMessageOnlineSend.getMsgId(), request.getSenderChannel());
    }


    /**
     * 发送消息给在线客户
     *
     * @param data
     * @param imMessageOnlineSend
     */
    private void sendMsgForOnline(ChatContentVo data, ImMessageOnlineSend imMessageOnlineSend) {
        Map<String /** ip **/, List<String /** client的主键ID:platform **/>> onlineIpClientMap =
                thousandChatCacheManager.findOnlineHostsByThousandGroupId(data.getToConversation());
        //  封装要推给接收方的消息
        WsResponse<ImMessageOnlineSend> responseModel = new WsResponse<>();
        responseModel.setCmd(WsResponseCmdEnum.ONLINE_MSG.getCmdCode());
        ApiResult<Boolean> result = ApiResult.result(ApiCode.SUCCESS);
        responseModel.setCode(result.getCode());
        responseModel.setMsg(result.getMessage());
        responseModel.setData(imMessageOnlineSend);
        responseModel.setReqId(null);

        onlineIpClientMap.forEach((ip, clientIdAndPlatforms) -> {
            channelSender.batchSendMsg(responseModel, ip, clientIdAndPlatforms);
        });
    }

    /**
     * 拼装发送消息体
     *
     * @param data
     * @param imClientSender
     * @param appId
     * @return
     */
    private ImMessageOnlineSend assembleImMessageOnlineSend(ChatContentVo data, ImClient imClientSender, Long appId) {
        // 生成消息id
        long messageId = SnowflakeUtil.getId();
        // 封装响应的实体
        ImMessageOnlineSend imMessageOnlineSend = new ImMessageOnlineSend();
        imMessageOnlineSend.setMsgId(messageId);
        imMessageOnlineSend.setSender(imClientSender.getClientId());
        Map<String, Object> content = JsonUtils.beanCopyDeep(data, Map.class);
        //action的属性无需要返回
        content.remove(BaseRequest.ACTION);
        imMessageOnlineSend.setContent(content);
        imMessageOnlineSend.setConversationId(data.getToConversation());
        imMessageOnlineSend.setCreateTime(new Date());
        imMessageOnlineSend.setWithdrawTime(null);
        imMessageOnlineSend.setWithdraw(false);
        imMessageOnlineSend.setEvent(false);
        imMessageOnlineSend.setSystemFlag(false);
        imMessageOnlineSend.setType(data.getType());
        imMessageOnlineSend.setAt(data.getAt());
        return imMessageOnlineSend;
    }

    /**
     * 响应发送方消息id等信息
     *
     * @param reqId
     * @param messageId
     * @param channel
     */
    private void response(String reqId, long messageId, Channel channel) {
        WsResponse<MsgVo> responseModel = new WsResponse<>();
        ApiResult<Boolean> result = ApiResult.result(ApiCode.SUCCESS);
        responseModel.setCmd(WsResponseCmdEnum.RES.getCmdCode());
        responseModel.setCode(result.getCode());
        responseModel.setMsg(result.getMessage());
        responseModel.setData(new MsgVo(messageId));
        responseModel.setReqId(reqId);
        // 响应发送方
        channelSender.sendMsgLocal((NioSocketChannel) channel, responseModel);
    }
}
