package com.wecloud.im.sdk.internal;

import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.wecloud.im.sdk.common.HttpMethod;
import com.wecloud.im.sdk.common.RequestMessage;
import com.wecloud.im.sdk.exception.WecloudImException;
import com.wecloud.im.sdk.model.AddOrModifyConversationAttributeRequest;
import com.wecloud.im.sdk.model.Conversation;
import com.wecloud.im.sdk.model.CreateConversationRequest;
import com.wecloud.im.sdk.model.ListConversationRequest;
import com.wecloud.im.sdk.model.SetConversationDoNotDisturbRequest;
import com.wecloud.im.sdk.model.SetConversationTopRequest;

import static com.wecloud.im.sdk.ResourcePathConstants.ADD_OR_MODIFY_CONVERSATION_ATTRIBUTE_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.CHAT_SETTING;
import static com.wecloud.im.sdk.ResourcePathConstants.CREATE_CONVERSATION_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.LIST_CONVERSATION_REQUEST;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotBlank;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotNull;

/**
 * client
 * @Author luozh
 * @Date 2022年04月13日 15:26
 * @Version 1.0
 */
public class WecloudImConversationOperation extends WecloudImOperation {

    public WecloudImConversationOperation(URL apiDomain, String appKey, String appSecret) {
        super(apiDomain, appKey, appSecret);
    }

    /**
     * 设置会话置顶
     * @Author luozh
     * @Date 2022年04月26日 03:57:51
     * @param setConversationTopRequest
     * @Return
     */
    public Boolean setConversationTop(SetConversationTopRequest setConversationTopRequest) {
        String userId = setConversationTopRequest.getUserId();
        Long conversationId = setConversationTopRequest.getConversationId();
        Boolean top = setConversationTopRequest.getTop();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        assertParameterNotNull(conversationId, "conversationId");
        assertParameterNotNull(top, "top");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("clientId", userId);
        param.put("conversationId", conversationId.toString());
        param.put("type", "TOP");
        param.put("on", top.toString());
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(CHAT_SETTING)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(setConversationTopRequest).build();

        doOperation(request);
        return true;
    }

    /**
     * 设置会话免打扰
     * @Author luozh
     * @Date 2022年04月26日 03:58:10
     * @param setConversationDoNotDisturbRequest 设置会话免打扰
     * @Return
     */
    public Boolean setConversationDoNotDisturb(SetConversationDoNotDisturbRequest setConversationDoNotDisturbRequest) {
        String userId = setConversationDoNotDisturbRequest.getUserId();
        Long conversationId = setConversationDoNotDisturbRequest.getConversationId();
        Boolean doNotDisturb = setConversationDoNotDisturbRequest.getDoNotDisturb();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        assertParameterNotNull(conversationId, "conversationId");
        assertParameterNotNull(doNotDisturb, "doNotDisturb");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("clientId", userId);
        param.put("conversationId", conversationId.toString());
        param.put("type", "DO_NOT_DISTURB");
        param.put("on", doNotDisturb.toString());
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(CHAT_SETTING)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(setConversationDoNotDisturbRequest).build();

        doOperation(request);
        return true;
    }

    /**
     * 会话列表
     * @Author luozh
     * @Date 2022年05月16日 09:43:38
     * @param listConversationRequest
     * @Return
     */
    public List<Conversation> listConversation(ListConversationRequest listConversationRequest) {
        // 参数校验
        assertParameterNotNull(listConversationRequest.getConversationIds(), "conversationIds");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("chatType", listConversationRequest.getChatType() + "");
        param.put("conversationIds", JSON.toJSONString(listConversationRequest.getConversationIds()));
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(LIST_CONVERSATION_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(listConversationRequest).build();

        Object result = doOperation(request);
        TypeReference<List<Conversation>> typeReference = new TypeReference<List<Conversation>>() {
        };
        List<Conversation> conversations = JSON.parseObject(JSON.toJSONString(result), typeReference);
        return conversations;
    }

    /**
     * 创建会话
     * @Author Shadow
     * @Date 2022年05月31日 04:26:58
     * @param createConversationRequest 创建会话请求
     * @Return 会话id
     */
    public Long create(CreateConversationRequest createConversationRequest) {
        // 参数校验
        assertParameterNotBlank(createConversationRequest.getName(), "name");
        assertParameterNotBlank(createConversationRequest.getCreatorUserId(), "creatorUserId");
        assertParameterNotBlank(createConversationRequest.getMemberUserIds(), "memberUserIds");
        assertParameterNotBlank(createConversationRequest.getChatType(), "chatType");
        // 校验chatType
        String[] allowType = new String[]{"1", "5", "6"};
        if (!Arrays.asList(allowType).contains(createConversationRequest.getChatType())) {
            throw new WecloudImException(500, "chatType 只能为 1 5 6");
        }
        // 校验通过 构建参数
        TypeReference<Map<String, Object>> typeReference = new TypeReference<Map<String, Object>>() {
        };
        Map<String, Object> param = JSONObject.parseObject(JSON.toJSONString(createConversationRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(CREATE_CONVERSATION_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(createConversationRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Long.class);
    }

    public Boolean addOrModifyConversationAttribute(AddOrModifyConversationAttributeRequest addOrModifyConversationAttributeRequest) {
        // 参数校验
        assertParameterNotNull(addOrModifyConversationAttributeRequest.getConversationId(), "conversationId");
        if (addOrModifyConversationAttributeRequest.getAttribute() == null) {
            addOrModifyConversationAttributeRequest.setAttribute(new HashMap<>());
        }
        Map<String, Object> param = new HashMap<>();
        param.put("conversationId", addOrModifyConversationAttributeRequest.getConversationId());
        param.put("attribute", addOrModifyConversationAttributeRequest.getAttribute());

        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(ADD_OR_MODIFY_CONVERSATION_ATTRIBUTE_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(addOrModifyConversationAttributeRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Boolean.class);

    }
}
