package com.wecloud.im.sdk.internal;

import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.wecloud.im.sdk.common.HttpMethod;
import com.wecloud.im.sdk.common.RequestMessage;
import com.wecloud.im.sdk.enums.GroupChatSettingTypeEnum;
import com.wecloud.im.sdk.model.CreateGroupRequest;
import com.wecloud.im.sdk.model.DismissGroupRequest;
import com.wecloud.im.sdk.model.GroupSettingRequest;
import com.wecloud.im.sdk.model.JoinGroupRequest;
import com.wecloud.im.sdk.model.LeaveGroupRequest;
import com.wecloud.im.sdk.model.ListGroupMembersRequest;
import com.wecloud.im.sdk.model.MutedGroupMemberRequest;

import static com.wecloud.im.sdk.ResourcePathConstants.CREATE_GROUP_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.DISMISS_GROUP_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.GROUP_SETTING_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.JOIN_GROUP_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.LEAVE_GROUP_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.LIST_GROUP_MEMBERS_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.MUTED_GROUP_MEMBER_REQUEST;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotBlank;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotNull;

/**
 * 群组操作
 * @Author luozh
 * @Date 2022年05月10日 09:30
 * @Version 1.0
 */
public class WecloudImGroupOperation extends WecloudImOperation {

    public WecloudImGroupOperation(URL apiDomain, String appKey, String appSecret) {
        super(apiDomain, appKey, appSecret);
    }

    /**
     * 创建群组
     * @Author luozh
     * @Date 2022年05月10日 10:27:41
     * @param createGroupRequest
     * @Return
     */
    public Long createGroup(CreateGroupRequest createGroupRequest) {
        String userId = createGroupRequest.getUserId();
        String memberIds = createGroupRequest.getMemberIds();
        // 必填参数校验
        assertParameterNotBlank(userId, "userId");
        assertParameterNotBlank(memberIds, "memberIds");
        // 校验通过 构建参数
        TypeReference<Map<String, String>> typeReference = new TypeReference<Map<String, String>>() {
        };
        Map<String, String> param = JSONObject.parseObject(JSON.toJSONString(createGroupRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(CREATE_GROUP_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(createGroupRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Long.class);
    }

    /**
     * 解散群组
     * @Author luozh
     * @Date 2022年05月10日 10:27:54
     * @param dismissGroupRequest
     * @Return
     */
    public Boolean dismissGroup(DismissGroupRequest dismissGroupRequest) {
        String userId = dismissGroupRequest.getUserId();
        String groupId = dismissGroupRequest.getGroupId();
        // 必填参数校验
        assertParameterNotBlank(userId, "userId");
        assertParameterNotBlank(groupId, "groupId");
        // 校验通过 构建参数
        TypeReference<Map<String, String>> typeReference = new TypeReference<Map<String, String>>() {
        };
        Map<String, String> param = JSONObject.parseObject(JSON.toJSONString(dismissGroupRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(DISMISS_GROUP_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(dismissGroupRequest).build();
        doOperation(request);
        return true;
    }

    /**
     * 加入群组
     * @Author luozh
     * @Date 2022年05月10日 10:28:07
     * @param joinGroupRequest
     * @Return
     */
    public Integer joinGroup(JoinGroupRequest joinGroupRequest) {
        String groupId = joinGroupRequest.getGroupId();
        String userIds = joinGroupRequest.getUserIds();
        // 必填参数校验
        assertParameterNotBlank(groupId, "groupId");
        assertParameterNotBlank(userIds, "userIds");
        // 校验通过 构建参数
        TypeReference<Map<String, String>> typeReference = new TypeReference<Map<String, String>>() {
        };
        Map<String, String> param = JSONObject.parseObject(JSON.toJSONString(joinGroupRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(JOIN_GROUP_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(joinGroupRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Integer.class);
    }

    /**
     * 离开群组
     * @Author luozh
     * @Date 2022年05月10日 10:28:21
     * @param leaveGroupRequest
     * @Return
     */
    public Integer leaveGroup(LeaveGroupRequest leaveGroupRequest) {
        String groupId = leaveGroupRequest.getGroupId();
        String userIds = leaveGroupRequest.getUserIds();
        // 必填参数校验
        assertParameterNotBlank(groupId, "groupId");
        assertParameterNotBlank(userIds, "userIds");
        // 校验通过 构建参数
        TypeReference<Map<String, String>> typeReference = new TypeReference<Map<String, String>>() {
        };
        Map<String, String> param = JSONObject.parseObject(JSON.toJSONString(leaveGroupRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(LEAVE_GROUP_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(leaveGroupRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Integer.class);
    }

    /**
     * 获取群成员列表
     * @Author luozh
     * @Date 2022年05月10日 10:28:34
     * @param listGroupMembersRequest
     * @Return
     */
    public List<String> listGroupMembers(ListGroupMembersRequest listGroupMembersRequest) {
        String groupId = listGroupMembersRequest.getGroupId();
        // 必填参数校验
        assertParameterNotBlank(groupId, "groupId");
        // 校验通过 构建参数
        TypeReference<Map<String, String>> typeReference = new TypeReference<Map<String, String>>() {
        };
        Map<String, String> param = JSONObject.parseObject(JSON.toJSONString(listGroupMembersRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(LIST_GROUP_MEMBERS_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(listGroupMembersRequest).build();
        Object result = doOperation(request);
        TypeReference<List<String>> resultType = new TypeReference<List<String>>() {
        };
        return JSON.parseObject(JSON.toJSONString(result), resultType);
    }


    /**
     * 群设置
     * @Author luozh
     * @Date 2022年05月12日 05:41:21
     * @param groupSettingRequest
     * @Return
     */
    public Boolean groupSetting(GroupSettingRequest groupSettingRequest) {
        // 参数校验
        GroupChatSettingTypeEnum type = groupSettingRequest.getType();
        String groupId = groupSettingRequest.getGroupId();
        Boolean on = groupSettingRequest.getOn();
        String name = groupSettingRequest.getName();
        String headPortrait = groupSettingRequest.getHeadPortrait();
        String operatorClientId = groupSettingRequest.getOperatorUserId();
        assertParameterNotBlank(groupId, "groupId");
        assertParameterNotNull(type, "type");
        assertParameterNotBlank(operatorClientId, "operatorClientId");
        switch (type) {
            case FORBID_SEND_PIC:
            case FORBID_SEND_LINK:
            case FORBID_ADD_FRIEND:
            case MUTED:
                assertParameterNotNull(on, "on");
                break;
            case NAME:
                assertParameterNotBlank(name, "name");
                break;
            case HEAD_PORTRAIT:
                assertParameterNotBlank(headPortrait, "headPortrait");
                break;
            default:
                break;
        }

        // 校验完成，构建参数
        Map<String, String> param = new HashMap<>();
        param.put("type", type.name());
        param.put("groupId", groupId);
        param.put("operatorClientId", operatorClientId);
        switch (type) {
            case FORBID_SEND_PIC:
            case FORBID_SEND_LINK:
            case FORBID_ADD_FRIEND:
            case MUTED:
                param.put("on", on.toString());
                break;
            case NAME:
                param.put("name", name);
                break;
            case HEAD_PORTRAIT:
                param.put("headPortrait", headPortrait);
                break;
            default:
                break;
        }

        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(GROUP_SETTING_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(groupSettingRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Boolean.class);
    }

    public Boolean mutedGroupMember(MutedGroupMemberRequest mutedGroupMemberRequest) {

        String groupId = mutedGroupMemberRequest.getGroupId();
        Boolean muted = mutedGroupMemberRequest.getMuted();
        String userIds = mutedGroupMemberRequest.getUserIds();
        // 必填参数校验
        assertParameterNotBlank(groupId, "groupId");
        assertParameterNotNull(muted, "muted");
        assertParameterNotBlank(userIds, "userIds");
        // 校验通过 构建参数
        Map<String, String> param = new HashMap<>();
        param.put("groupId", groupId);
        param.put("muted", muted.toString());
        param.put("userIds", userIds);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(MUTED_GROUP_MEMBER_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(mutedGroupMemberRequest).build();
        Object result = doOperation(request);
        return JSON.parseObject(JSON.toJSONString(result), Boolean.class);
    }
}
