package com.wecloud.im.friend.controller;

import com.wecloud.im.entity.ImClient;
import com.wecloud.im.enums.FriendStateEnum;
import com.wecloud.im.friend.param.ImCreateRecommendParam;
import com.wecloud.im.friend.param.ImDeleteFriendParam;
import com.wecloud.im.friend.param.ImDeleteRecommendParam;
import com.wecloud.im.friend.param.ImFriendApplyDto;
import com.wecloud.im.friend.param.ImFriendApplyParam;
import com.wecloud.im.friend.param.ImFriendApproveParam;
import com.wecloud.im.friend.param.ImFriendBaseDto;
import com.wecloud.im.friend.param.ImFriendBaseParam;
import com.wecloud.im.friend.param.ImFriendPageParam;
import com.wecloud.im.friend.param.ImFriendRecommendDto;
import com.wecloud.im.friend.service.ImFriendService;
import com.wecloud.im.service.ImClientService;
import io.geekidea.springbootplus.framework.common.api.ApiCode;
import io.geekidea.springbootplus.framework.common.api.ApiResult;
import io.geekidea.springbootplus.framework.common.controller.BaseController;
import io.geekidea.springbootplus.framework.common.enums.BaseEnum;
import io.geekidea.springbootplus.framework.core.pagination.Paging;
import io.geekidea.springbootplus.framework.log.annotation.OperationLog;
import io.geekidea.springbootplus.framework.log.enums.OperationLogType;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * @author lixiaozhong
 * 好友与好友推荐相关接口
 */
@Slf4j
@RestController
@RequestMapping("/friend")
@Api(value = "好友与好友推荐相关API", tags = {"好友关系"})
public class ImFriendController extends BaseController {

    @Autowired
    private ImFriendService imFriendService;

    @Autowired
    private ImClientService imClientService;

    /**
     * 查询好友信息，只有自己的好友才查得到
     */
    @PostMapping("/info")
    @ApiOperation(value = "查询好友信息，只有自己的好友才查得到")
    public ApiResult<ImFriendApplyDto> getFriendInfo(@RequestBody ImFriendBaseParam param) {
        if(param == null) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
        }
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        ImClient friendClient = imClientService.getCacheImClient(currentClient.getFkAppid(), param.getFriendClientId());
        if(friendClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        ImFriendApplyDto friendInfo = imFriendService.getFriendInfo(currentClient.getClientId(), friendClient.getClientId());
        return ApiResult.ok(friendInfo);
    }

    /**
     * 申请添加好友
     */
    @PostMapping("/apply")
    @ApiOperation(value = "申请添加好友")
    public ApiResult<Boolean> applyFriend(@RequestBody ImFriendApplyParam param) {
        if(param == null) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
        }
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        ImClient friendClient = imClientService.getCacheImClient(currentClient.getFkAppid(), param.getFriendClientId());
        if(friendClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        if(currentClient.getId().equals(friendClient.getId())) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
        }
        imFriendService.applyFriend(currentClient, friendClient, param.getFriendName(), param.getRequestRemark());
        return ApiResult.ok();
    }

    /**
     * 待接受的好友请求列表，最多只返回1000个
     */
    @PostMapping("/unsureFriends")
    @ApiOperation(value = "待接受的好友请求列表，最多只返回1000个")
    public ApiResult<List<ImFriendApplyDto>> getUnsureFriends() {
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        List<ImFriendApplyDto> friendInfo = imFriendService.getUnsureFriends(currentClient);
        return ApiResult.ok(friendInfo);
    }

    /**
     * 接受/拒绝好友申请
     */
    @PostMapping("/approve")
    @ApiOperation(value = "接受/拒绝好友申请")
    public ApiResult<Boolean> approveFriend(@RequestBody ImFriendApproveParam param) {
        if(param == null || param.getAgree() == null) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
        }
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        ImClient friendClient = imClientService.getCacheImClient(currentClient.getFkAppid(), param.getFriendClientId());
        if(friendClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }

        boolean ok = imFriendService.approveFriend(currentClient, friendClient, param.getAgree(), param.getRejectRemark());
        return ApiResult.ok(ok);
    }


    /**
     * 批量删除好友
     */
    @PostMapping("/batchDelete")
    @ApiOperation(value = "删除好友")
    public ApiResult<Boolean> batchDeleteFriend(@RequestBody ImDeleteFriendParam param) {
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }

        imFriendService.batchDeleteFriend(currentClient, param.getFriendClientIds());
        return ApiResult.ok();
    }

    /**
     * 好友分页列表
     */
    @PostMapping("/getPageList")
    @OperationLog(name = "好友分页列表", type = OperationLogType.PAGE)
    @ApiOperation(value = "好友分页列表")
    public ApiResult<Paging<ImFriendBaseDto>> getImFriendPageList(@RequestBody ImFriendPageParam pageParam) {
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }

        Paging<ImFriendBaseDto> friends = imFriendService.getImFriendPageList(currentClient.getId(), pageParam);
        return ApiResult.ok(friends);
    }

    /**
     * 批量创建好友推荐
     */
    @PostMapping("/recommend/batchCreate")
    @OperationLog(name = "批量创建好友推荐", type = OperationLogType.PAGE)
    @ApiOperation(value = "批量创建好友推荐")
    public ApiResult<Boolean> batchCreateRecommend(@RequestBody ImCreateRecommendParam param) {
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        if(CollectionUtils.isEmpty(param.getRecommendFriends())) {
            return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
        }
        for (ImFriendRecommendDto p : param.getRecommendFriends()) {
            if (BaseEnum.valueOf(FriendStateEnum.class, p.getSource()) == null) {
                return ApiResult.fail(ApiCode.PARAMETER_EXCEPTION, null);
            }
        }
        imFriendService.batchCreateRecommend(currentClient, param.getRecommendFriends());
        return ApiResult.ok();
    }

    /**
     * 好友推荐分页列表
     */
    @PostMapping("/recommend/getPageList")
    @OperationLog(name = "好友推荐分页列表", type = OperationLogType.PAGE)
    @ApiOperation(value = "好友推荐分页列表")
    public ApiResult<Paging<ImFriendRecommendDto>> getImFriendRecommendPageList(@RequestBody ImFriendPageParam pageParam) {
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        Paging<ImFriendRecommendDto> res = imFriendService.getImFriendRecommendPageList(currentClient.getId(), pageParam);
        return ApiResult.ok(res);

    }

    /**
     * 批量删除好友推荐
     */
    @PostMapping("/recommend/batchDelete")
    @ApiOperation(value = "删除好友推荐")
    public ApiResult<Boolean> batchDeleteRecommend(@RequestBody ImDeleteRecommendParam param) {
        ImClient currentClient = imClientService.getCurentClient();
        if(currentClient == null) {
            return ApiResult.fail(ApiCode.CLIENT_NOT_FOUNT, null);
        }
        imFriendService.batchDeleteRecommend(currentClient, param.getFriendClientIds());
        return ApiResult.ok();
    }
}

