package com.wecloud.im.thousandchat.action;

import io.geekidea.springbootplus.framework.common.api.ApiCode;
import io.geekidea.springbootplus.framework.common.api.ApiResult;
import io.netty.channel.Channel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.RequestBody;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.wecloud.dispatch.annotation.ActionMapping;
import com.wecloud.dispatch.common.BaseRequest;
import com.wecloud.dispatch.extend.ActionRequest;
import com.wecloud.im.entity.ImApplication;
import com.wecloud.im.entity.ImClient;
import com.wecloud.im.entity.ImConversationMembers;
import com.wecloud.im.entity.ImMessage;
import com.wecloud.im.entity.ImMessageOnlineSend;
import com.wecloud.im.mq.MqSender;
import com.wecloud.im.param.ChatContentVo;
import com.wecloud.im.param.ImClientSimpleDto;
import com.wecloud.im.param.ImConversationQueryVo;
import com.wecloud.im.param.MsgVo;
import com.wecloud.im.service.ImApplicationService;
import com.wecloud.im.service.ImClientService;
import com.wecloud.im.service.ImConversationMembersService;
import com.wecloud.im.service.ImConversationService;
import com.wecloud.im.service.ImMessageService;
import com.wecloud.im.thousandchat.cache.ThousandChatCacheManager;
import com.wecloud.im.thousandchat.param.LastestReceivedMsg;
import com.wecloud.im.thousandchat.service.ThousandChatService;
import com.wecloud.im.ws.enums.WsResponseCmdEnum;
import com.wecloud.im.ws.model.WsResponse;
import com.wecloud.im.ws.sender.ChannelSender;
import com.wecloud.pushserver.client.model.constant.MqConstant;
import com.wecloud.pushserver.client.model.dto.PushDTO;
import com.wecloud.utils.JsonUtils;
import com.wecloud.utils.SnowflakeUtil;

/**
 * @Description 处理Cmd请求
 * 抽象类 策略设计模式
 * @Author hewei hwei1233@163.com
 * @Date 2020-01-02
 */
@Slf4j
@Component
@ActionMapping
@Api(value = "万人群消息处理", tags = {"万人群消息处理"})
public class ThousandChatAction {

    @Autowired
    private ImApplicationService imApplicationService;
    @Autowired
    private ImClientService imClientService;
    @Autowired
    private ImConversationService imConversationService;
    @Autowired
    private ImConversationMembersService imConversationMembersService;
    @Autowired
    private ImMessageService imMessageService;
    @Autowired
    private ChannelSender channelSender;
    @Autowired
    private ThousandChatCacheManager thousandChatCacheManager;
    @Autowired
    private ThousandChatService thousandChatService;

    @Autowired
    private MqSender mqSender;

    @ActionMapping("/chat/thousand/send")
    @ApiOperation(value = "万人群消息发送")
    public void sendMsg(ActionRequest request, ChatContentVo data, String reqId) {
        if (log.isDebugEnabled()) {
            log.debug("接收到参数，reqId: {},\n data: {}, ", data);
        }

        //查看接收的群属性，是否万人群
        ImConversationQueryVo conversation = imConversationService.getCacheImConversationById(data.getToConversation());

        if (conversation == null) {
            log.warn("会reqId: {} ,会话id: {}db中不存在", reqId, data.getToConversation());
            return;
        }

        // 查询发送者client
        ImClient imClientSender = imClientService.getCacheImClient(request.getSenderClientId());
        if (imClientSender == null) {
            log.warn("根据senderClientId: {} 查找不到 imClientSender！", request.getSenderClientId());
            return;
        }

        // 查询imApplication
        ImApplication imApplication = imApplicationService.getCacheById(imClientSender.getFkAppid());
        if (imApplication == null) {
            log.warn("根据appId: {} 查找不到 imApplication！", imClientSender.getFkAppid());
            return;
        }

        ImMessageOnlineSend imMessageOnlineSend = assembleImMessageOnlineSend(data, imClientSender, imApplication.getId());

        // 在线用户直接发消息
        sendMsgForOnline(data, imMessageOnlineSend);

        // 给所有人（在线+离线）遍历发送
        // 先查询该会话所有成员
        List<ImConversationMembers> membersList = imConversationMembersService.list(
                new QueryWrapper<ImConversationMembers>().lambda()
                        .eq(ImConversationMembers::getFkConversationId, data.getToConversation())
        );
        // 再给所有人发 todo 需要改成批量
        for (ImConversationMembers conversationMembers : membersList) {
            // 入库 保存收件箱
//            long imInboxId = SnowflakeUtil.getId();
//            saveImInbox(imApplication, toConversationId, messageId, conversationMembers, imInboxId);

            // 查询接收方
            ImClient imClientReceiver = imClientService.getOne(new QueryWrapper<ImClient>().lambda()
                    .eq(ImClient::getFkAppid, imApplication.getId())
                    .eq(ImClient::getId, conversationMembers.getFkClientId()));
            if (imClientReceiver == null) {
                continue;
            }
            // 异步推送系统通知消息
            PushDTO pushDTO = mqSender.buildPushDto(data.getPush(), imClientReceiver.getId(), imClientReceiver.getClientId(), imApplication);
            if (pushDTO != null) {
                mqSender.orderSend(MqConstant.Topic.IM_ORDER_MSG_TOPIC, MqConstant.Tag.IM_ORDER_MSG_TAG, pushDTO);
            }
        }

        // 响应发送方消息id等信息
        response(reqId, imMessageOnlineSend.getMsgId(), request.getSenderChannel());

    }

    /**
     * 消息修改为已接收状态
     */
    @ActionMapping("/imState/msgReceivedUpdate")
    @ApiOperation(value = "万人群消息修改为已接收状态")
    public ApiResult<Boolean> updateImMsgReceived(@RequestBody @Validated LastestReceivedMsg lastestReceivedMsg) {
        return thousandChatService.updateImMsgReceived(lastestReceivedMsg);
    }

    /**
     * 消息修改为已读状态
     */
    @ActionMapping("/imState/msgReadUpdate")
    @ApiOperation(value = "万人群消息修改为已读状态")
    public ApiResult<Boolean> updateInMsgReadUpdate(Long lastestMsgId) {
        return thousandChatService.updateImMsgRead(lastestMsgId);
    }

    /**
     * 发送消息给在线客户
     * @param data
     * @param imMessageOnlineSend
     */
    private void sendMsgForOnline(ChatContentVo data, ImMessageOnlineSend imMessageOnlineSend) {
        Map<String /** ip **/, List<String /** client的主键ID:platform **/>> onlineIpClientMap =
                thousandChatCacheManager.findOnlineHostsByThousandGroupId(data.getToConversation());
        //  封装要推给接收方的消息
        WsResponse<ImMessageOnlineSend> responseModel = new WsResponse<>();
        responseModel.setCmd(WsResponseCmdEnum.ONLINE_MSG.getCmdCode());
        ApiResult<Boolean> result = ApiResult.result(ApiCode.SUCCESS);
        responseModel.setCode(result.getCode());
        responseModel.setMsg(result.getMessage());
        responseModel.setData(imMessageOnlineSend);
        responseModel.setReqId(null);

        onlineIpClientMap.forEach((ip, clientIdAndPlatforms) -> {
            channelSender.batchSendMsg(responseModel, ip, clientIdAndPlatforms);
        });
    }

    /**
     * 拼装发送消息体
     * @param data
     * @param imClientSender
     * @param appId
     * @return
     */
    private ImMessageOnlineSend assembleImMessageOnlineSend(ChatContentVo data, ImClient imClientSender, Long appId) {
        // 生成消息id
        long messageId = SnowflakeUtil.getId();
        // 入库  保存消息至消息表
        ImClientSimpleDto client = new ImClientSimpleDto().setId(imClientSender.getId()).setFkAppid(appId);
        ImMessage imMessage = imMessageService.saveImMessage(client, data);

        // 封装响应的实体
        ImMessageOnlineSend imMessageOnlineSend = new ImMessageOnlineSend();
        imMessageOnlineSend.setMsgId(imMessage.getId());
        imMessageOnlineSend.setSender(imClientSender.getClientId());
        Map<String, Object> content = JsonUtils.beanCopyDeep(data, Map.class);
        //action的属性无需要返回
        content.remove(BaseRequest.ACTION);
        imMessageOnlineSend.setContent(content);
        imMessageOnlineSend.setConversationId(data.getToConversation());
        imMessageOnlineSend.setCreateTime(imMessage.getCreateTime());
        imMessageOnlineSend.setWithdrawTime(imMessage.getWithdrawTime());
        imMessageOnlineSend.setWithdraw(imMessage.getWithdraw());
        imMessageOnlineSend.setEvent(imMessage.getEvent());
        imMessageOnlineSend.setSystemFlag(imMessage.getSystemFlag());
        imMessageOnlineSend.setType(data.getType());
        imMessageOnlineSend.setAt(imMessage.getAt());
        return imMessageOnlineSend;
    }

    /**
     * 响应发送方消息id等信息
     *
     * @param reqId
     * @param messageId
     * @param channel
     */
    private void response(String reqId, long messageId, Channel channel) {
        WsResponse<MsgVo> responseModel = new WsResponse<>();
        ApiResult<Boolean> result = ApiResult.result(ApiCode.SUCCESS);
        responseModel.setCmd(WsResponseCmdEnum.RES.getCmdCode());
        responseModel.setCode(result.getCode());
        responseModel.setMsg(result.getMessage());
        responseModel.setData(new MsgVo(messageId));
        responseModel.setReqId(reqId);
        // 响应发送方
        channelSender.sendMsgLocal((NioSocketChannel)channel, responseModel);
    }
}
