package com.wecloud.im.netty.handler;


import com.alibaba.fastjson.JSONObject;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.wecloud.im.netty.core.WsReadHandler;
import com.wecloud.im.ws.model.WsConstants;
import com.wecloud.im.ws.service.MangerChannelService;
import com.wecloud.im.ws.utils.FullHttpRequestUtils;
import io.geekidea.springbootplus.config.constant.CommonConstant;
import io.geekidea.springbootplus.framework.common.api.ApiCode;
import io.geekidea.springbootplus.framework.common.api.ApiResult;
import io.geekidea.springbootplus.framework.shiro.jwt.JwtToken;
import io.geekidea.springbootplus.framework.shiro.service.ShiroLoginService;
import io.geekidea.springbootplus.framework.shiro.util.JwtUtil;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.http.FullHttpRequest;
import io.netty.handler.codec.http.HttpResponseStatus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;

/**
 * @Description 聊天模块 http请求处理
 * @Author hewei hwei1233@163.com
 * @Date 2019-07-19
 */
@Component
@Slf4j
public class NettyApiRequest {

    @Autowired
    private ShiroLoginService shiroLoginService;

    @Resource
    private MangerChannelService appUserChannelsService;

    @Resource
    private WsReadHandler appImReadHandler;

    /**
     * http请求接收
     *
     * @param ctx
     * @param msg
     * @param httpRequest
     * @throws Exception
     */
    public void handle(ChannelHandlerContext ctx, Object msg, FullHttpRequest httpRequest) throws Exception {
        if (!(msg instanceof FullHttpRequest)) {
            String context = new JsonMapper().writeValueAsString(ApiResult.fail());
            FullHttpRequestUtils.send(ctx, context, HttpResponseStatus.OK);
            return;
        }
        String path = httpRequest.uri();
        String body = FullHttpRequestUtils.getBody(httpRequest);

        if (log.isDebugEnabled()) {
            log.info("httpRequest:\n" + httpRequest.toString() + "\n" + body);
        }

        if (path.contains(WsConstants.WS_URL)) {
            /*
              app聊天http升级webSocket
             */
            this.initWs(ctx, httpRequest);

        }
    }

    /**
     * 初始化websocket
     */
    private void initWs(ChannelHandlerContext ctx, FullHttpRequest httpRequest) throws Exception {
        Map<String, String> paramMap = FullHttpRequestUtils.parameterParse(httpRequest);
        String token = paramMap.get(WsConstants.TOKEN);

        DecodedJWT jwtInfo = JwtUtil.getJwtInfo(token);
        String payload = jwtInfo.getPayload();
        Base64.Decoder decoder = Base64.getDecoder();
        payload = new String(decoder.decode(payload), StandardCharsets.UTF_8);
        JSONObject jsonObject = JSONObject.parseObject(payload);
        String appKey = (String) jsonObject.get("appKey");
        String clientId = (String) jsonObject.get(CommonConstant.CLIENT_ID);

        if (StringUtils.isBlank(token)) {
            String context = new JsonMapper().writeValueAsString(ApiResult.result(ApiCode.FAIL, "token不能为空", (Object) null));
            FullHttpRequestUtils.send(ctx, context, HttpResponseStatus.OK);
            return;
        }
        if (JwtUtil.isExpired(token)) {
            String context = new JsonMapper().writeValueAsString(ApiResult.result(ApiCode.FAIL, "JWT Token已过期,token", (Object) null));
            FullHttpRequestUtils.send(ctx, context, HttpResponseStatus.OK);
            return;
        }

        // 从redis获取jwt的token 验签token
        JwtToken jwtToken = shiroLoginService.getTJwtTokenForRedis(token);

        if (jwtToken == null) {
            log.info("jwtToken == null ,token和redis不一致, clientId:" + clientId + ",token:" + token);
            String context = new JsonMapper().writeValueAsString(ApiResult.result(ApiCode.FAIL, "jwtToken == null ,token和redis不一致", (Object) null));
            FullHttpRequestUtils.send(ctx, context, HttpResponseStatus.OK);
            return;
        }

        if ((!jwtToken.getClientId().equals(clientId)) || (!jwtToken.getAppKey().equals(appKey))) {
            String context = new JsonMapper().writeValueAsString(ApiResult.result(ApiCode.FAIL, "clientId  appKey 不一致", (Object) null));
            FullHttpRequestUtils.send(ctx, context, HttpResponseStatus.OK);
            log.info("clientId  appKey 不一致");
            return;
        }

        // 设置uri前缀
        httpRequest.setUri(WsConstants.WS_URL);

        // 保持当前连接
        ctx.fireChannelRead(httpRequest.retain());

        // 设置属性值 userid - channel
        ctx.channel().attr(MangerChannelService.CLIENT_ID).set(clientId);
        ctx.channel().attr(MangerChannelService.APP_KEY).set(appKey);
        ctx.channel().attr(MangerChannelService.READ_IDLE_TIMES).set(0);// 读空闲的计数=0

//         添加长连接handler
        ctx.pipeline().addLast("appImHandler", appImReadHandler);

        // 保存用户上下文对象
        appUserChannelsService.online(appKey, clientId, (NioSocketChannel) ctx.channel());

        //移除当前api处理handler， 不再参与长连接处理
        ctx.pipeline().remove("SingleHttpRequestHandler");

    }


}
