package com.wecloud.im.biz.module.message.controller;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.Arrays;
import java.util.List;

import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.discovery.DiscoveryClient;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.alibaba.fastjson.JSON;
import com.wecloud.can.core.util.ApiResult;
import com.wecloud.im.biz.module.message.controller.param.GroupOwnerTransferParam;
import com.wecloud.im.biz.module.message.controller.param.ModifyGroupAdminParam;
import com.wecloud.im.biz.module.message.controller.param.group.CreateGroupParam;
import com.wecloud.im.biz.module.message.controller.param.group.DismissGroupParam;
import com.wecloud.im.biz.module.message.controller.param.group.JoinGroupParam;
import com.wecloud.im.biz.module.message.controller.param.group.LeaveGroupParam;
import com.wecloud.im.biz.module.message.controller.param.group.ListGroupMembersParam;
import com.wecloud.im.biz.module.message.controller.param.group.MutedMembersParam;
import com.wecloud.im.biz.module.message.service.ImGroupService;

/**
 * 群聊接口
 * @Author luozh
 * @Date 2022年05月10日 14:14
 * @Version 1.0
 */
@AllArgsConstructor
@Slf4j
@RestController
@RequestMapping("/group")
public class ImGroupController {

    private final ImGroupService groupService;

    private final DiscoveryClient discoveryClient;

    /**
     * 创建群组
     * @Author luozh
     * @Date 2022年05月10日 02:16:19
     * @Return
     */
    @PostMapping("/create")
    public ApiResult<Long> createGroup(@RequestBody CreateGroupParam param) {
        log.info("创建群组请求 参数: {}", JSON.toJSONString(param));
        List<String> memberClientIds = Arrays.asList(param.getMemberIds().split(","));
        Long conversationId = groupService.createGroup(param.getUserId(), param.getGroupName(), memberClientIds);
        return ApiResult.ok(conversationId);
    }

    /**
     * 解散群组
     * @Author luozh
     * @Date 2022年05月10日 02:16:19
     * @Return
     */
    @PostMapping("/dismiss")
    public ApiResult<Boolean> dismissGroup(@RequestBody DismissGroupParam param) {
        log.info("解散群组请求 参数: {}", JSON.toJSONString(param));
        return ApiResult.ok(groupService.dismissGroup(param.getUserId(), param.getGroupId()));
    }

    /**
     * 加入群组
     * @Author luozh
     * @Date 2022年05月10日 02:16:19
     * @Return
     */
    @PostMapping("/join")
    public ApiResult<Integer> joinGroup(@Validated @RequestBody JoinGroupParam param) {
        log.info("加入群组请求 参数: {}", JSON.toJSONString(param));
        List<String> memberIds = Arrays.asList(param.getUserIds().split(","));
        return ApiResult.ok(groupService.joinGroup(param.getInviterUserId(), param.getGroupId(), memberIds));
    }

    /**
     * 离开群组
     * @Author luozh
     * @Date 2022年05月10日 02:16:19
     * @Return
     */
    @PostMapping("/leave")
    public ApiResult<Integer> leaveGroup(@RequestBody LeaveGroupParam param) {
        log.info("离开群组请求 参数: {}", JSON.toJSONString(param));
        List<String> memberIds = Arrays.asList(param.getUserIds().split(","));
        return ApiResult.ok(groupService.leaveGroup(param.getOperatorUserId(), param.getGroupId(), memberIds));
    }

    /**
     *
     * @Author luozh
     * @Date 2022年05月10日 02:16:19
     * @Return
     * @return
     */
    @PostMapping("/member/list")
    public ApiResult<List<String>> listGroupMembers(@RequestBody ListGroupMembersParam param) {
        log.info("获取群组成员请求 参数: {}", JSON.toJSONString(param));
        return ApiResult.ok(groupService.listGroupMembers(param.getGroupId()));
    }

    /**
     * 禁言群成员
     * @Author luozh
     * @Date 2022年05月10日 02:16:19
     * @Return
     * @return
     */
    @PostMapping("/member/muted")
    public ApiResult<Boolean> mutedMembers(@RequestBody MutedMembersParam param) {
        log.info("禁言群成员请求 参数: {}", JSON.toJSONString(param));
        return ApiResult.ok(groupService.mutedMembers(param));
    }

    /**
     * 群主转移
     * @Author luozh
     * @Date 2022年05月17日 04:20:26
     * @param param
     * @Return
     */
    @PostMapping("/ownerTransfer")
    public ApiResult<Boolean> groupOwnerTransfer(@RequestBody GroupOwnerTransferParam param) {
        log.info("群主转移请求 参数: {}", JSON.toJSONString(param));
        return ApiResult.ok(groupService.groupOwnerTransfer(param.getGroupId(), param.getNewGroupOwnerUserId()));
    }

    /**
     * 添加群管理员
     * @Author luozh
     * @Date 2022年05月17日 06:24:26
     * @param param
     * @Return
     */
    @PostMapping("/addAdmin")
    public ApiResult<Boolean> addGroupAdmin(@Validated @RequestBody ModifyGroupAdminParam param) {
        log.info("添加群管理员请求 参数: {}", JSON.toJSONString(param));
        return ApiResult.ok(groupService.addGroupAdmin(param.getGroupId(), param.getAdminUserIds()));
    }

    /**
     * 移除群管理员
     * @Author luozh
     * @Date 2022年05月17日 06:24:40
     * @param param
     * @Return
     */
    @PostMapping("/removeAdmin")
    public ApiResult<Boolean> removeGroupAdmin(@Validated @RequestBody ModifyGroupAdminParam param) {
        log.info("移除群管理员请求 参数: {}", JSON.toJSONString(param));
        return ApiResult.ok(groupService.removeGroupAdmin(param.getGroupId(), param.getAdminUserIds()));
    }

    @GetMapping("/getServiceInstance")
    public List<ServiceInstance> getServiceInstance() {
        List<ServiceInstance> serviceInstanceList = discoveryClient.getInstances("providers:com.wecloud.im.router.RouterSendService::");
        return serviceInstanceList;
    }
}
