package com.wecloud.im.mq;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.google.common.collect.Lists;
import com.wecloud.im.entity.ImApplication;
import com.wecloud.im.entity.ImClient;
import com.wecloud.im.entity.ImClientDevice;
import com.wecloud.im.service.ImClientDeviceService;
import com.wecloud.im.service.ImInboxService;
import com.wecloud.im.ws.model.request.PushVO;
import com.wecloud.pushserver.client.model.dto.ApplicationDTO;
import com.wecloud.pushserver.client.model.dto.ClientDTO;
import com.wecloud.pushserver.client.model.dto.ClientDeviceDTO;
import com.wecloud.pushserver.client.model.dto.MessageDTO;
import com.wecloud.pushserver.client.model.dto.PushDTO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.rocketmq.client.producer.SendResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;

import static com.wecloud.im.ws.ImConstant.PUSH_BODY;
import static com.wecloud.im.ws.ImConstant.PUSH_TITLE;

/**
 * @Author wenzhida
 * @Date 2022/1/28 11:47
 * @Description mq推送服务
 */
@Slf4j
@Component
public class MqSender {

    @Autowired
    private ImInboxService imInboxService;

    @Autowired
    private RocketMqProducerService rocketMqProducerService;

    @Autowired
    private ImClientDeviceService imClientDeviceService;

    /**
     * mq同步推送方法
     *
     * @param topic
     * @param tag
     * @param pushDTO
     * @return
     */
    public SendResult synSend(String topic, String tag, PushDTO pushDTO) {
        log.info("mq同步推送topic: {} tag: {} 推送内容: {}", topic, tag, JSON.toJSONString(pushDTO));
        SendResult sendResult = rocketMqProducerService.synSend(topic, tag, JSON.toJSONString(pushDTO));
        log.info("mq同步推送topic: {} tag: {} 返回结果: {}", topic, tag, JSON.toJSONString(sendResult));
        return sendResult;
    }

    /**
     * mq异步推送方法
     *
     * @param topic
     * @param tag
     * @param pushDTO
     * @return
     */
    public void sendAsync(String topic, String tag, PushDTO pushDTO) {
        log.info("mq异步推送topic: {} tag: {} 推送内容: {}", topic, tag, JSON.toJSONString(pushDTO));
        rocketMqProducerService.sendAsyncDefault(topic, tag, JSON.toJSONString(pushDTO));
    }

    /**
     * mq按顺序发送
     *
     * @param topic
     * @param tag
     * @param pushDTO
     * @return
     */
    public SendResult orderSend(String topic, String tag, PushDTO pushDTO) {
        log.info("mq按顺序发送topic: {} tag: {} 推送内容: {}", topic, tag, JSON.toJSONString(pushDTO));
        SendResult sendResult = rocketMqProducerService.orderSend(topic, tag, JSON.toJSONString(pushDTO), 1);
        log.info("mq按顺序发送topic: {} tag: {} 返回结果: {}", topic, tag, JSON.toJSONString(sendResult));
        return sendResult;
    }


    public PushDTO buildPushDto(PushVO pushVO, ImClient imClientReceiver, ImApplication imApplication) {
        PushDTO pushDTO = new PushDTO();
        MessageDTO messageDTO = new MessageDTO();
        if (pushVO == null) {
            pushVO = new PushVO();
            pushVO.setTitle(PUSH_TITLE);
            pushVO.setSubTitle(PUSH_BODY);
        }
        messageDTO.setTitle(pushVO.getTitle());
        messageDTO.setSubTitle(pushVO.getSubTitle());
        messageDTO.setData(pushVO.getData());
        // 统计未读消息数量
        int badge = imInboxService.countMyNotReadCount(imClientReceiver.getId());
        messageDTO.setBadge(badge);
        ClientDTO clientDTO = new ClientDTO();
        clientDTO.setClientId(imClientReceiver.getClientId());
        List<ImClientDevice> clientDeviceList = imClientDeviceService.list(
                new QueryWrapper<ImClientDevice>().lambda()
                        .eq(ImClientDevice::getFkClientId, imClientReceiver.getId()));
        if (CollectionUtils.isEmpty(clientDeviceList)) {
            log.info("接收人信息 {} 查无推送设备", imClientReceiver.getId());
            return null;
        }
        List<ClientDeviceDTO> deviceDTOList = Lists.newArrayList();
        for (ImClientDevice imClientDevice : clientDeviceList) {
            ClientDeviceDTO clientDeviceDTO = new ClientDeviceDTO();
            clientDeviceDTO.setValid(imClientDevice.getValid());
            clientDeviceDTO.setDeviceType(imClientDevice.getDeviceType());
            clientDeviceDTO.setDeviceToken(imClientDevice.getDeviceToken());
            deviceDTOList.add(clientDeviceDTO);
        }
        clientDTO.setDeviceDTOList(deviceDTOList);
        messageDTO.setClientDTO(clientDTO);
        ApplicationDTO applicationDTO = new ApplicationDTO();
        applicationDTO.setId(imApplication.getId());
        applicationDTO.setAppKey(imApplication.getAppKey());
        applicationDTO.setAppSecret(imApplication.getAppSecret());
        applicationDTO.setAppName(imApplication.getAppName());
        applicationDTO.setIosPushChannel(imApplication.getIosPushChannel());
        applicationDTO.setAndroidPushChannel(imApplication.getAndroidPushChannel());
        applicationDTO.setUmengKey(imApplication.getUmengKey());
        applicationDTO.setUmengSecret(imApplication.getUmengSecret());
        applicationDTO.setFirebaseSecret(imApplication.getFirebaseSecret());
        messageDTO.setApplicationDTO(applicationDTO);
        pushDTO.setMessageDTOList(Lists.newArrayList(messageDTO));
        return pushDTO;
    }
}
