package com.wecloud.im.sdk.internal;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.wecloud.im.sdk.common.HttpMethod;
import com.wecloud.im.sdk.common.RequestMessage;
import com.wecloud.im.sdk.model.*;

import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.wecloud.im.sdk.ResourcePathConstants.*;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotBlank;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotNull;

/**
 * 通讯录Operation
 * @Author luozh
 * @Date 2022年05月16日 09:35
 * @Version 1.0
 */
public class WecloudImContactsOperation extends WecloudImOperation {

    public WecloudImContactsOperation(URL apiDomain, String appKey, String appSecret) {
        super(apiDomain, appKey, appSecret);
    }

    /**
     * 待接受的好友请求列表
     * @Author luozh
     * @Date 2022年04月21日 05:08:13
     * @param getUnsureFriendsRequest
     * @Return
     */
    public List<UnsureFriend> getUnsureFriends(GetUnsureFriendsRequest getUnsureFriendsRequest) {
        String userId = getUnsureFriendsRequest.getUserId();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId + "");

        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(GET_UNSURE_FRIENDS)
                .setMethod(HttpMethod.GET).setParameters(param)
                .setOriginalRequest(getUnsureFriendsRequest).build();

        Object result = doOperation(request);
        TypeReference<List<UnsureFriend>> typeReference = new TypeReference<List<UnsureFriend>>() {
        };
        List<UnsureFriend> unsureFriends = JSON.parseObject(JSON.toJSONString(result), typeReference);
        return unsureFriends;
    }

    /**
     * 获取好友列表(全量)
     * @Author luozh
     * @Date 2022年04月28日 03:01:21
     * @param getFriendsRequest
     * @Return
     */
    public List<Friend> getFriends(GetFriendsRequest getFriendsRequest) {
        String userId = getFriendsRequest.getUserId();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId + "");
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(GET_FRIENDS)
                .setMethod(HttpMethod.GET).setParameters(param)
                .setOriginalRequest(getFriendsRequest).build();

        Object result = doOperation(request);
        TypeReference<List<Friend>> typeReference = new TypeReference<List<Friend>>() {
        };
        List<Friend> unsureFriends = JSON.parseObject(JSON.toJSONString(result), typeReference);
        return unsureFriends;
    }

    /**
     * 好友请求
     * @Author luozh
     * @Date 2022年04月28日 03:01:47
     * @param friendRequestRequest
     */
    public Boolean friendRequest(FriendRequestRequest friendRequestRequest) {
        String userId = friendRequestRequest.getUserId();
        String friendId = friendRequestRequest.getFriendId();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        assertParameterNotNull(friendId, "friendId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("clientId", userId + "");
        param.put("friendClientId", friendId + "");
        param.put("friendName", Optional.ofNullable(friendRequestRequest.getFriendName()).orElse(""));
        param.put("requestRemark", Optional.ofNullable(friendRequestRequest.getRequestRemark()).orElse(""));
        param.put("needVerify", friendRequestRequest.getNeedVerify());
        param.put("addFriendScene",friendRequestRequest.getAddFriendScene());
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(FRIEND_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(friendRequestRequest).build();
        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;
    }

    /**
     * 群成员好友请求
     * @Author luozh
     * @Date 2022年05月16日 09:44:19
     * @param groupMembersFriendRequestRequest
     * @Return
     */
    public Boolean groupMembersFriendRequest(GroupMembersFriendRequestRequest groupMembersFriendRequestRequest) {
        Long conversationId = groupMembersFriendRequestRequest.getConversationId();
        String userId = groupMembersFriendRequestRequest.getUserId();
        String friendId = groupMembersFriendRequestRequest.getFriendId();
        // 参数校验
        assertParameterNotNull(conversationId, "conversationId");
        assertParameterNotNull(userId, "userId");
        assertParameterNotNull(friendId, "friendId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("conversationId", conversationId + "");
        param.put("clientId", userId + "");
        param.put("friendClientId", friendId + "");
        param.put("friendName", Optional.ofNullable(groupMembersFriendRequestRequest.getFriendName()).orElse(""));
        param.put("requestRemark", Optional.ofNullable(groupMembersFriendRequestRequest.getRequestRemark()).orElse(""));
        param.put("needVerify", groupMembersFriendRequestRequest.getNeedVerify());
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(GROUP_MEMBERS_FRIEND_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(groupMembersFriendRequestRequest).build();
        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;
    }

    /**
     * 分页获取黑名单
     * @Author luozh
     * @Date 2022年05月16日 09:43:16
     * @param pageBlacklistRequest
     * @Return
     */
    public PageResult<Blacklist> pageBlacklist(PageBlacklistRequest pageBlacklistRequest) {
        // 参数校验
        assertParameterNotNull(pageBlacklistRequest.getUserId(), "userId");
        // 校验通过 构建参数
        if (pageBlacklistRequest.getPageIndex() == null) {
            pageBlacklistRequest.setPageIndex(1);
        }
        if (pageBlacklistRequest.getPageSize() == null) {
            pageBlacklistRequest.setPageSize(10);
        }
        Map<String, Object> param = new HashMap<>();
        param.put("clientId", pageBlacklistRequest.getUserId());
        param.put("pageIndex", pageBlacklistRequest.getPageIndex() + "");
        param.put("pageSize", pageBlacklistRequest.getPageSize() + "");
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(PAGE_BLACKLIST_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(pageBlacklistRequest).build();

        Object result = doOperation(request);
        TypeReference<PageResult<Blacklist>> typeReference = new TypeReference<PageResult<Blacklist>>() {
        };
        PageResult<Blacklist> blacklistPage = JSON.parseObject(JSON.toJSONString(result), typeReference);
        return blacklistPage;
    }


    public Boolean unfriend(UnfriendRequest unfriendRequest) {
        String userId = unfriendRequest.getUserId();
        String friendUserIds = unfriendRequest.getFriendUserIds();
        // 参数校验
        assertParameterNotBlank(userId, "userId");
        assertParameterNotBlank(friendUserIds, "friendUserIds");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);
        param.put("friendUserIds", friendUserIds);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(UNFRIEND_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(unfriendRequest).build();

        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;

    }

    public Boolean remarkFriend(RemarkFriendRequest remarkFriendRequest) {

        String userId = remarkFriendRequest.getUserId();
        String friendUserId = remarkFriendRequest.getFriendUserId();
        String remark = remarkFriendRequest.getRemark();
        // 参数校验
        assertParameterNotBlank(userId, "userId");
        assertParameterNotBlank(friendUserId, "friendUserId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);
        param.put("friendUserId", friendUserId);
        param.put("friendRemark", remark);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(REMARK_FRIEND_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(remarkFriendRequest).build();

        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;
    }
}
