/*
 * Copyright 2019-2029 geekidea(https://github.com/geekidea)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.geekidea.springbootplus;

import com.alibaba.cloud.nacos.NacosDiscoveryProperties;
import com.alibaba.cloud.nacos.NacosServiceManager;
import com.alibaba.cloud.nacos.discovery.NacosWatch;
import com.alibaba.nacos.api.naming.PreservedMetadataKeys;
import com.wecloud.im.register.GetIpUtils;
import io.geekidea.springbootplus.framework.util.PrintApplicationInfo;
import org.mybatis.spring.annotation.MapperScan;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.ServletComponentScan;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.cloud.client.discovery.EnableDiscoveryClient;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;
import org.springframework.transaction.annotation.EnableTransactionManagement;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * spring-boot-plus 项目启动入口
 *
 * @author geekidea
 * @since 2018-11-08
 */
@EnableDiscoveryClient
@EnableAsync
@EnableScheduling
@EnableTransactionManagement
@EnableConfigurationProperties
@ServletComponentScan
@EnableCaching
//@ConfigurationPropertiesScan("com.wecloud.im")
@MapperScan({"io.geekidea.springbootplus.**.mapper", "com.wecloud.**.mapper"})
@SpringBootApplication(scanBasePackages = {"io.geekidea.springbootplus", "com.wecloud"})
public class SpringBootPlusApplication {

    @Value("${netty.port}")
    private String nettyPort;
    @Resource
    private GetIpUtils getIpUtils;

    public static void main(String[] args) {
        // 启动spring-boot-plus
        ConfigurableApplicationContext context = SpringApplication.run(SpringBootPlusApplication.class, args);
        // 打印项目信息
        PrintApplicationInfo.print(context);
        // 打印项目提示
        PrintApplicationInfo.printTip(context);
    }

    @Bean
    @ConditionalOnMissingBean
    public NacosDiscoveryProperties nacosProperties() {
        return new NacosDiscoveryProperties();
    }

    @Bean
    @ConditionalOnMissingBean
    @ConditionalOnProperty(value = {"spring.cloud.nacos.discovery.watch.enabled"}, matchIfMissing = true)
    public NacosWatch nacosWatch(NacosServiceManager nacosServiceManager, NacosDiscoveryProperties nacosDiscoveryProperties, ObjectProvider<ThreadPoolTaskScheduler> taskScheduler) {
        Map<String, String> metadataMap = nacosDiscoveryProperties.getMetadata();
        if (metadataMap == null) {
            metadataMap = new HashMap<>();
        }
        String key = "ip.netty.public";
        metadataMap.put("startup.time", new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()));
        String value = getIpUtils.getPublicIp() + ":" + nettyPort;
        metadataMap.put(key, value);
        // 设置心跳的周期，单位为秒，这里将心跳间隔设置为3秒：
        metadataMap.put(PreservedMetadataKeys.HEART_BEAT_INTERVAL, "3000");
        // 设置心跳超时时间，单位为秒，这里将心跳超时时间设为6秒，
        // 即服务端6秒收不到客户端心跳，会将该客户端注册的实例设为不健康：
        metadataMap.put(PreservedMetadataKeys.HEART_BEAT_TIMEOUT, "6000");
        // 设置实例删除的超时时间，单位为秒，这里将实例删除超时时间设为9秒，
        // 即服务端9秒收不到客户端心跳，会将该客户端注册的实例删除：
        metadataMap.put(PreservedMetadataKeys.IP_DELETE_TIMEOUT, "9000");

        nacosDiscoveryProperties.setMetadata(metadataMap);
        return new NacosWatch(nacosServiceManager, nacosDiscoveryProperties, taskScheduler);
    }

}
