package com.wecloud.im.ws.manager;

import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.util.AttributeKey;
import lombok.extern.slf4j.Slf4j;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.stereotype.Component;

import com.wecloud.im.ws.cache.UserStateListener;
import com.wecloud.im.ws.model.ClientInfo;
import com.wecloud.im.ws.utils.RedisUtils;

/**
 * channel内容管理，在线、离线等信息在channel里
 * @author lixiaozhong
 */
@Component
@Slf4j
public class ChannelManager {

    /**
     * client端对应channel的数据
     * key: appId-clientId-platform
     * value: channel对象
     */
    public static final Map<String, ClientInfo> SESSION_INFO_MAP = new ConcurrentHashMap<>();

    /**
     * CLIENT_ID,是客户端的字符串id
     */
    public static final AttributeKey<Long> CLIENT_ID = AttributeKey.valueOf("ci");

    public static final AttributeKey<Long> APPLICATION_ID = AttributeKey.valueOf("ai");

    public static final AttributeKey<Integer> READ_IDLE_TIMES = AttributeKey.valueOf("readIdleTimes");

    /**
     * LANGUAGE
     */
    AttributeKey<String> LANGUAGE = AttributeKey.valueOf("la");

    /**
     * APP_VERSION
     */
    AttributeKey<String> APP_VERSION = AttributeKey.valueOf("av");
    AttributeKey<String> TOKEN = AttributeKey.valueOf("to");
    AttributeKey<String> DEVICEID = AttributeKey.valueOf("dc");
    public static final AttributeKey<Integer> PLATFORM = AttributeKey.valueOf("pt");

    /**
     * client上线
     * userID绑定channel
     * @param clientId
     * @param platform
     * @param channel
     */
    public void online(Long clientId, Integer platform, NioSocketChannel channel) {
        String longChannelId = channel.id().asLongText();
        this.putSessionInfoMap(clientId, platform, channel);

        UserStateListener.triggerOnlineEvent(clientId, platform, longChannelId);
    }

    /**
     * 下线移除channel
     *
     * @param clientId
     * @param platform
     * @param channelHandlerContext
     */
    public void offline(Long clientId, Integer platform, ChannelHandlerContext channelHandlerContext) {

        String longChannelId = channelHandlerContext.channel().id().asLongText();

        // 关掉连接
        channelHandlerContext.close();


        // 移除本地维护的channel
        delSessionInfoMap(clientId, platform);

        UserStateListener.triggerOfflineEvent(clientId, platform, longChannelId);

    }

    public static String genKeyForSessionInfoMap(Long clientId, Integer platform) {
        return clientId + RedisUtils.SPLIT + platform;
    }

    private void putSessionInfoMap(Long clientId, Integer platform, NioSocketChannel channel) {
        ClientInfo clientInfo = new ClientInfo();
        clientInfo.setDeviceId("");
        clientInfo.setNioSocketChannel(channel);
        clientInfo.setToken("");
        ChannelManager.SESSION_INFO_MAP.put(genKeyForSessionInfoMap(clientId, platform), clientInfo);
    }

    private void delSessionInfoMap(Long clientId, Integer platform) {
        ChannelManager.SESSION_INFO_MAP.remove(genKeyForSessionInfoMap(clientId, platform));
    }

}
