package com.jumeirah.common.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.jumeirah.common.convert.SysUserConvert;
import com.jumeirah.common.entity.MerchantRole;
import com.jumeirah.common.entity.MerchantUser;
import com.jumeirah.common.enums.StateEnum;
import com.jumeirah.common.mapper.MerchantUserMapper;
import com.jumeirah.common.param.LoginParam;
import com.jumeirah.common.param.MerchantUserPageParam;
import com.jumeirah.common.service.MerchantRolePermissionService;
import com.jumeirah.common.service.MerchantRoleService;
import com.jumeirah.common.service.MerchantUserService;
import com.jumeirah.common.vo.LoginSysUserTokenVo;
import com.jumeirah.common.vo.MerchantUserQueryForAppVo;
import com.jumeirah.common.vo.MerchantUserQueryVo;
import io.geekidea.springbootplus.config.properties.JwtProperties;
import io.geekidea.springbootplus.config.properties.SpringBootPlusProperties;
import io.geekidea.springbootplus.framework.common.api.ApiCode;
import io.geekidea.springbootplus.framework.common.api.ApiResult;
import io.geekidea.springbootplus.framework.common.service.impl.BaseServiceImpl;
import io.geekidea.springbootplus.framework.core.pagination.PageInfo;
import io.geekidea.springbootplus.framework.core.pagination.Paging;
import io.geekidea.springbootplus.framework.shiro.cache.MerchantLoginRedisService;
import io.geekidea.springbootplus.framework.shiro.jwt.JwtToken;
import io.geekidea.springbootplus.framework.shiro.util.JwtTokenUtil;
import io.geekidea.springbootplus.framework.shiro.util.JwtUtil;
import io.geekidea.springbootplus.framework.shiro.util.SaltUtil;
import io.geekidea.springbootplus.framework.shiro.vo.LoginUserVo;
import io.geekidea.springbootplus.framework.util.PasswordUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.subject.Subject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.http.HttpServletResponse;
import java.time.Duration;
import java.util.Set;

/**
 * 商家 服务实现类
 *
 * @author wei
 * @since 2020-09-28
 */
@Slf4j
@Service
public class MerchantUserServiceImpl extends BaseServiceImpl<MerchantUserMapper, MerchantUser> implements MerchantUserService {

    @Lazy
    @Autowired
    private SpringBootPlusProperties springBootPlusProperties;

    @Lazy
    @Autowired
    private RedisTemplate redisTemplate;
    @Lazy
    @Autowired
    private MerchantLoginRedisService merchantLoginRedisService;

    @Lazy
    @Autowired
    private JwtProperties jwtProperties;

    @Autowired
    private MerchantUserMapper merchantUserMapper;

    @Autowired
    private MerchantRoleService merchantRoleService;

    @Autowired
    private MerchantRolePermissionService merchantRolePermissionService;

    public MerchantUser getSysUserByUsername(String username) throws Exception {
        MerchantUser sysUser = new MerchantUser().setUsername(username);
        return merchantUserMapper.selectOne(new QueryWrapper<MerchantUser>(sysUser));
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public ApiResult<LoginSysUserTokenVo> login(LoginParam loginParam, HttpServletResponse response, String language) throws Exception {

//        // 校验验证码
//        checkVerifyCode(loginParam.getVerifyToken(), loginParam.getCode());

        String username = loginParam.getUsername();
        // 从数据库中获取登录用户信息
        MerchantUser merchantUser = getSysUserByUsername(username);
        if (merchantUser == null) {

            log.error("登录失败,loginParam:{}", loginParam);
//            throw new AuthenticationException("用户名或密码错误");
            return ApiResult.fail(ApiCode.PWD_OR_USERNAME_ERROR, language);
        }
        if (StateEnum.DISABLE.getCode().equals(merchantUser.getState())) {
            throw new AuthenticationException("账号已禁用");
        }

        // 实际项目中，前端传过来的密码应先加密
        // 原始密码明文：123456
        // 原始密码前端加密：sha256(123456)
        // 后台加密规则：sha256(sha256(123456) + salt)
        String encryptPassword = PasswordUtil.encrypt(loginParam.getPassword(), merchantUser.getSalt());
        if (!encryptPassword.equals(merchantUser.getPassword())) {
            return ApiResult.fail(ApiCode.PWD_OR_USERNAME_ERROR, language);
        }

        // 将系统用户对象转换成登录用户对象
        LoginUserVo loginSysUserVo = SysUserConvert.INSTANCE.merchantUserToLoginSysUserVo(merchantUser);

//        // 获取部门
//        SysDepartment sysDepartment = sysDepartmentService.getById(merchantUser.getDepartmentId());
//        if (sysDepartment == null) {
//            throw new AuthenticationException("部门不存在");
//        }
//        if (!StateEnum.ENABLE.getCode().equals(sysDepartment.getState())) {
//            throw new AuthenticationException("部门已禁用");
//        }
//        loginSysUserVo.setDepartmentId(sysDepartment.getId())
//                .setDepartmentName(sysDepartment.getName());

        // 获取当前用户角色
        Long roleId = merchantUser.getRoleId();
        MerchantRole merchantRole = merchantRoleService.getById(roleId);
        if (merchantRole == null) {
            throw new AuthenticationException("角色不存在");
        }
        if (StateEnum.DISABLE.getCode().equals(merchantRole.getState())) {
            throw new AuthenticationException("角色已禁用");
        }
        loginSysUserVo.setRoleId(merchantRole.getId())
                .setRoleName(merchantRole.getName())
                .setRoleCode(merchantRole.getCode());

        // 获取当前用户权限
        Set<String> permissionCodes = merchantRolePermissionService.getPermissionCodesByRoleId(roleId);
//        if (CollectionUtils.isEmpty(permissionCodes)) {
//            throw new AuthenticationException("权限列表不能为空");
//        }
        loginSysUserVo.setPermissionCodes(permissionCodes);

        // 获取数据库中保存的盐值
        String newSalt = SaltUtil.getSalt(merchantUser.getSalt(), jwtProperties);

        // 生成token字符串并返回
        Long expireSecond = jwtProperties.getExpireSecond();
        String token = JwtUtil.generateToken(username, newSalt, Duration.ofSeconds(expireSecond));
        log.debug("token:{}", token);

        // 创建AuthenticationToken
        JwtToken jwtToken = JwtToken.build(token, username,merchantUser.getId(), newSalt, expireSecond, "mer");

        boolean enableShiro = springBootPlusProperties.getShiro().isEnable();
        if (enableShiro) {
            // 从SecurityUtils里边创建一个 subject
            Subject subject = SecurityUtils.getSubject();
            // 执行认证登录
            subject.login(jwtToken);
        } else {
            log.warn("未启用Shiro");
        }

        // 缓存登录信息到Redis
        merchantLoginRedisService.cacheLoginInfo(jwtToken, loginSysUserVo);
        log.debug("登录成功,username:{}", username);

/*        // 缓存登录信息到redis
        String tokenSha256 = DigestUtils.sha256Hex(token);
        redisTemplate.opsForValue().set(tokenSha256, loginSysUserVo, 1, TimeUnit.DAYS);*/

        // 返回token和登录用户信息对象
        LoginSysUserTokenVo loginSysUserTokenVo = new LoginSysUserTokenVo();
        loginSysUserTokenVo.setToken(token);
        loginSysUserTokenVo.setLoginSysUserVo(loginSysUserVo);

        // 设置token响应头
        response.setHeader(JwtTokenUtil.getTokenName(), loginSysUserTokenVo.getToken());

        return ApiResult.ok(loginSysUserTokenVo);
    }


    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveMerchantUser(MerchantUser merchantUser) throws Exception {
        return super.save(merchantUser);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateMerchantUser(MerchantUser merchantUser) throws Exception {
        return super.updateById(merchantUser);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean deleteMerchantUser(Long id) throws Exception {
        return super.removeById(id);
    }

    @Override
    public MerchantUserQueryVo getMerchantUserById(Long id) throws Exception {
        return merchantUserMapper.getMerchantUserById(id);
    }

    @Override
    public Paging<MerchantUserQueryVo> getMerchantUserPageList(MerchantUserPageParam merchantUserPageParam) throws Exception {
        Page<MerchantUserQueryVo> page = new PageInfo<>(merchantUserPageParam, OrderItem.desc(getLambdaColumn(MerchantUser::getCreateTime)));
        IPage<MerchantUserQueryVo> iPage = merchantUserMapper.getMerchantUserPageList(page, merchantUserPageParam);
        return new Paging<MerchantUserQueryVo>(iPage);
    }

    @Override
    public Paging<MerchantUserQueryForAppVo> getMerchantUserPageListForApp(MerchantUserPageParam merchantUserPageParam) throws Exception {
        Page<MerchantUserQueryForAppVo> page = new PageInfo<>(merchantUserPageParam, OrderItem.desc(getLambdaColumn(MerchantUser::getCreateTime)));
        IPage<MerchantUserQueryForAppVo> iPage = merchantUserMapper.getMerchantUserForAppPageList(page, merchantUserPageParam);
        return new Paging<MerchantUserQueryForAppVo>(iPage);
    }

}
