package com.wecloud.im.ws.cache;

import com.wecloud.im.ws.model.redis.ClientChannelInfo;
import com.wecloud.im.ws.utils.InitIp;
import com.wecloud.im.ws.utils.RedisUtils;
import com.wecloud.im.ws.utils.SpringBeanUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * @author hewei123@163.com
 * @Description 用户与redis绑定
 * @createTime 2020年04月14日 16:21:00
 */
@Service
@Slf4j
public class UserStateCacheManager extends UserStateListener {

    /**
     * 维护client的多端数据:
     * 值为set集合,netty的channel的id
     */
    private static final String CLIENTS = "cis:";
    /**
     * 设备类型   web,ios,android,ios,other
     */
    private static final String DEVICE_TYPE = "ty";

    /**
     * 用户公网ip，在公网部署集群需要用到
     */
    private static final String PUBLIC_IP = "pip";
    /**
     * 推送token
     */
    private static final String DEVICE_TOKEN = "dt";
    /**
     * 设备ID
     */
    private static final String DEVICE_ID = "di";

    @Autowired
    private RedisUtils redisUtils;

    @Override
    public void onLineEvent(String appKey, String clientId, Integer platform, String longChannelId) {
        log.info("ws用户上线保存redis连接ip: {}, uid: {}", InitIp.lAN_IP, longChannelId);
        // 先删除旧的重复的platform
        Set<String> platformAndIps = redisUtils.getForSetMembers(getUserStateCacheKey(appKey, clientId));
        for(String platformAndIp : platformAndIps) {
            String[] split = platformAndIp.split(RedisUtils.SPLIT);
            String innerPlatform = split[0];
            if(innerPlatform.equals(String.valueOf(platform))) {
                redisUtils.removeForSet(getUserStateCacheKey(appKey, clientId), platformAndIp);
            }
        }

        redisUtils.addForSet(getUserStateCacheKey(appKey, clientId), platform + RedisUtils.SPLIT + InitIp.lAN_IP, 10, TimeUnit.DAYS);
    }

    @Override
    public void offlineEvent(String appKey, String clientId, Integer platform, String longChannelId) {
        log.info("ws用户离线删除redis key,uid:" + longChannelId);
        redisUtils.removeForSet(getUserStateCacheKey(appKey, clientId), platform + RedisUtils.SPLIT + InitIp.lAN_IP);
    }

    /**
     * 根据clientId获取在线用户信息
     * @param appKey
     * @param clientId
     * @return
     */
    public List<ClientChannelInfo> findOnlineInfosByClientId(String appKey, String clientId) {

        // 获取所有 CLIENTS的 <platform>:<ip>
        Set<String> platformAndIs = redisUtils.getForSetMembers(getUserStateCacheKey(appKey, clientId));
        ArrayList<ClientChannelInfo> clientChannelInfos = new ArrayList<>();

        for(String platformAndIp : platformAndIs) {
            String[] split = platformAndIp.split(RedisUtils.SPLIT);
            ClientChannelInfo clientChannelInfo = new ClientChannelInfo();
            clientChannelInfo.setPlatform(Integer.valueOf(split[0]));
            clientChannelInfo.setLanIp(split[1]);
            clientChannelInfos.add(clientChannelInfo);
        }

        return clientChannelInfos;
    }

    /**
     * 判断用户是否在线
     * @param appKey
     * @param clientId
     * @return true表示在线，false表示离线
     */
    public boolean isOnline(String appKey, String clientId) {
        Set<String> platformAndIs = redisUtils.getForSetMembers(getUserStateCacheKey(appKey, clientId));
        return platformAndIs.size() > 0;
    }

    private String getUserStateCacheKey(String appKey, String clientId) {
        return CLIENTS + appKey + RedisUtils.SPLIT + clientId;
    }

}
