package com.wecloud.rtc.service.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.wecloud.im.ws.utils.RedisUtils;
import com.wecloud.rtc.entity.redis.RtcChannelInfo;
import com.wecloud.rtc.entity.redis.RtcRedisKey;
import com.wecloud.rtc.service.MangerRtcCacheService;
import com.wecloud.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;

@Service
public class MangerRtcCacheServiceImpl implements MangerRtcCacheService {

    @Autowired
    private RedisUtils redisUtils;


    @Override
    public boolean delChannelInfo(Long rtcChannelId) {
        String channelKey = String.format(RtcRedisKey.RTC_CHANNEL_INFO, rtcChannelId);
        redisUtils.delKey(channelKey);
        return true;
    }

    @Override
    public boolean channelIsEmpty(Long rtcChannelId) {
        List<String> clientListByRtcChannelId = getClientListByRtcChannelId(rtcChannelId);

//        // 移除自己
//        clientListByRtcChannelId.remove(appKey + clientId);
        return clientListByRtcChannelId.isEmpty();
    }

    @Override
    public RtcChannelInfo getRtcChannelInfo(Long rtcChannelId) throws JsonProcessingException {
        // 频道信息
        String channelKey = String.format(RtcRedisKey.RTC_CHANNEL_INFO, rtcChannelId);
        String value = redisUtils.getKey(channelKey);

        if (StringUtils.isBlank(value)) {
            return null;
        }

        return JsonUtils.decodeJson(value, RtcChannelInfo.class);
    }


    @Override
    public void create(Long clientId, Long toClientId, Long rtcChannelId) {
        // --- 频道信息
        RtcChannelInfo rtcChannelInfo = new RtcChannelInfo();

        //当前房主
        rtcChannelInfo.setOwner(clientId);
        //创建时间
        rtcChannelInfo.setCreateTimestamp(System.currentTimeMillis());

        String rtcChannelInfoJson = JsonUtils.encodeJson(rtcChannelInfo);

        // --- 保存频道信息
        String channelKey = String.format(RtcRedisKey.RTC_CHANNEL_INFO, rtcChannelId);
        redisUtils.addKey(channelKey, rtcChannelInfoJson, Duration.ofDays(10));

        // 用户当前在线的频道ID
        String userJoinChannelKey = String.format(RtcRedisKey.USER_JOIN_CHANNEL, clientId);
        redisUtils.addKey(userJoinChannelKey, rtcChannelId.toString(), Duration.ofDays(10));

        // 频道中存在的用户
        String rtcChannelUsers = String.format(RtcRedisKey.RTC_CHANNEL_USERS, rtcChannelId);
        redisUtils.addForSet(rtcChannelUsers, clientId + "", 10, TimeUnit.DAYS);
        redisUtils.addForSet(rtcChannelUsers, toClientId + "", 10, TimeUnit.DAYS);

    }

    @Override
    public void join(Long clientId, Long rtcChannelId) {

        //用户当前在线的频道ID
        String userJoinChannelKey = String.format(RtcRedisKey.USER_JOIN_CHANNEL, clientId);
        redisUtils.addKey(userJoinChannelKey, rtcChannelId.toString(), Duration.ofDays(10));
    }


    @Override
    public void leave(Long clientId, Long rtcChannelId) {

        //用户当前在线的频道ID
        String userJoinChannelKey = String.format(RtcRedisKey.USER_JOIN_CHANNEL, clientId);
        redisUtils.delKey(userJoinChannelKey);

        //频道中存在的用户
        String rtcChannelUsersKey = String.format(RtcRedisKey.RTC_CHANNEL_USERS, rtcChannelId);
        redisUtils.removeForSet(rtcChannelUsersKey, clientId + "");
    }

    @Override
    public List<String> getClientListByRtcChannelId(Long rtcChannelId) {

        //频道中存在的用户
        String key = String.format(RtcRedisKey.RTC_CHANNEL_USERS, rtcChannelId);

        Set<String> forSetMembers = redisUtils.getForSetMembers(key);

        return new ArrayList<>(forSetMembers);

    }


    @Override
    public Long getRtcChannelIdListByClientId(Long clientId) {

        //用户当前在线的频道ID
        String userJoinChannelKey = String.format(RtcRedisKey.USER_JOIN_CHANNEL, clientId);
        String key = redisUtils.getKey(userJoinChannelKey);
        if (StringUtils.isBlank(key)) {
            return null;
        }
        return Long.valueOf(key);
    }

    @Override
    public boolean getBusyStatus(Long clientId) {

        //用户当前在线的频道ID
        String userJoinChannelKey = String.format(RtcRedisKey.USER_JOIN_CHANNEL, clientId);

        String key = redisUtils.getKey(userJoinChannelKey);

        return key != null && !key.isEmpty();

    }
}
