package com.wecloud.im.sdk.internal;

import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.wecloud.im.sdk.common.HttpMethod;
import com.wecloud.im.sdk.common.RequestMessage;
import com.wecloud.im.sdk.model.Attributes;
import com.wecloud.im.sdk.model.BannedUserRequest;
import com.wecloud.im.sdk.model.ChatRoomMember;
import com.wecloud.im.sdk.model.ClientLoginRequest;
import com.wecloud.im.sdk.model.ClientRelation;
import com.wecloud.im.sdk.model.GetUserAttributesRequest;
import com.wecloud.im.sdk.model.ImClient;
import com.wecloud.im.sdk.model.ListChatRoomMemberRequest;
import com.wecloud.im.sdk.model.ModifyUserAttributesRequest;
import com.wecloud.im.sdk.model.ModifyUserRequest;
import com.wecloud.im.sdk.model.RegisterClientRequest;
import com.wecloud.im.sdk.model.Token;
import com.wecloud.im.sdk.model.UserRelationRequest;
import com.wecloud.im.sdk.model.UsersRelationRequest;

import static com.wecloud.im.sdk.ResourcePathConstants.BANNED_USER_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.GET_TOKEN_URL;
import static com.wecloud.im.sdk.ResourcePathConstants.GET_USER_ATTRIBUTES_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.LIST_CHAT_ROOM_MEMBER_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.MODIFY_USER_ATTRIBUTES_REQUEST;
import static com.wecloud.im.sdk.ResourcePathConstants.MODIFY_USER_URL;
import static com.wecloud.im.sdk.ResourcePathConstants.REGISTER_CLIENT_URL;
import static com.wecloud.im.sdk.ResourcePathConstants.USER_RELATION_REQUEST;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotBlank;
import static com.wecloud.im.sdk.utils.CodingUtils.assertParameterNotNull;

/**
 * client
 * @Author luozh
 * @Date 2022年04月13日 15:26
 * @Version 1.0
 */
public class WecloudImUserOperation extends WecloudImOperation {

    public WecloudImUserOperation(URL apiDomain, String appKey, String appSecret) {
        super(apiDomain, appKey, appSecret);
    }

    /**
     * 注册客户端
     * @Author luozh
     * @Date 2022年04月13日 03:38:58
     * @param registerClientRequest
     * @Return
     */
    public ImClient registerClient(RegisterClientRequest registerClientRequest) {
        String userId = registerClientRequest.getUserId();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("clientId", registerClientRequest.getUserId() + "");
        param.put("headPortrait", Optional.ofNullable(registerClientRequest.getHeadPortrait()).orElse(""));
        param.put("nickname", Optional.ofNullable(registerClientRequest.getNickname()).orElse(""));
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(REGISTER_CLIENT_URL)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(registerClientRequest).build();

        Object result = doOperation(request);
        ImClient imClient = JSON.parseObject(JSON.toJSONString(result), ImClient.class);
        return imClient;
    }

    /**
     * 获取用户token
     * @Author luozh
     * @Date 2022年04月18日 11:52:18
     * @param clientLoginRequest
     * @Return
     */
    public Token getToken(ClientLoginRequest clientLoginRequest) {
        String userId = clientLoginRequest.getUserId();
        Integer deviceType = clientLoginRequest.getDeviceType();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        assertParameterNotNull(deviceType, "deviceType");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("deviceType", clientLoginRequest.getDeviceType() + "");
        param.put("clientId", clientLoginRequest.getUserId() + "");

        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(GET_TOKEN_URL)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(clientLoginRequest).build();

        Object result = doOperation(request);
        Token token = JSON.parseObject(JSON.toJSONString(result), Token.class);
        return token;
    }

    /**
     * 修改用户
     * @Author luozh
     * @Date 2022年04月18日 03:31:04
     * @param modifyUserRequest
     * @Return
     */
    public Boolean modifyUser(ModifyUserRequest modifyUserRequest) {
        String userId = modifyUserRequest.getUserId();
        String headPortrait = modifyUserRequest.getHeadPortrait();
        String nickname = modifyUserRequest.getNickname();
        // 参数校验
        assertParameterNotNull(userId, "userId");
        assertParameterNotBlank(nickname, "nickname");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId + "");
        param.put("headPortrait", headPortrait);
        param.put("nickname", nickname);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(MODIFY_USER_URL)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(modifyUserRequest).build();

        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;
    }

    /**
     * 聊天室成员列表
     * @Author luozh
     * @Date 2022年05月16日 09:45:03
     * @param listChatRoomMemberRequest
     * @Return
     */
    public List<ChatRoomMember> listChatRoomMember(ListChatRoomMemberRequest listChatRoomMemberRequest) {
        // 参数校验
        assertParameterNotNull(listChatRoomMemberRequest.getChatRoomId(), "chatRoomId");
        // 校验通过 构建参数
        Map<String, Object> param = new HashMap<>();
        param.put("chatRoomId", listChatRoomMemberRequest.getChatRoomId() + "");
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(LIST_CHAT_ROOM_MEMBER_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(listChatRoomMemberRequest).build();

        Object result = doOperation(request);
        TypeReference<List<ChatRoomMember>> typeReference = new TypeReference<List<ChatRoomMember>>() {
        };
        List<ChatRoomMember> chatRoomMembers = JSON.parseObject(JSON.toJSONString(result), typeReference);
        return chatRoomMembers;
    }

    /**
     * 用户关系
     * @Author luozh
     * @Date 2022年05月16日 09:43:29
     * @param userRelationRequest
     * @Return
     */
    public ClientRelation userRelation(UserRelationRequest userRelationRequest) {
        String clientId = userRelationRequest.getUserId();
        String friendClientId = userRelationRequest.getFriendUserId();
        assertParameterNotBlank(clientId, "userId");
        assertParameterNotBlank(friendClientId, "friendUserId");

        Map<String, Object> param = new HashMap<>();
        param.put("clientId", clientId);
        param.put("friendClientId", friendClientId);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(USER_RELATION_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(userRelationRequest).build();

        Object result = doOperation(request);
        ClientRelation clientRelation = JSON.parseObject(JSON.toJSONString(result), ClientRelation.class);
        return clientRelation;
    }

    public List<ClientRelation> usersRelation(UsersRelationRequest userRelationRequest) {
        return null;
    }

    public Boolean bannedUser(BannedUserRequest bannedUserRequest) {

        String userId = bannedUserRequest.getUserId();
        Integer minute = bannedUserRequest.getMinute();
        assertParameterNotBlank(userId, "userId");

        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);
        param.put("minute", minute.toString());
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(BANNED_USER_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(bannedUserRequest).build();

        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;
    }

    public Attributes getUserAttributes(GetUserAttributesRequest getUserAttributesRequest) {
        String userId = getUserAttributesRequest.getUserId();
        assertParameterNotBlank(userId, "userId");

        Map<String, Object> param = new HashMap<>();
        param.put("userId", userId);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(GET_USER_ATTRIBUTES_REQUEST)
                .setMethod(HttpMethod.GET).setParameters(param)
                .setOriginalRequest(getUserAttributesRequest).build();

        Object result = doOperation(request);
        Attributes attributes = JSON.parseObject(JSON.toJSONString(result), Attributes.class);
        return attributes;
    }

    public Boolean modifyUserAttributes(ModifyUserAttributesRequest modifyUserAttributesRequest) {
        String userId = modifyUserAttributesRequest.getUserId();
        Attributes attributes = modifyUserAttributesRequest.getAttributes();
        assertParameterNotBlank(userId, "userId");
        assertParameterNotNull(attributes, "attributes");
        // 校验通过 构建参数
        TypeReference<Map<String, Object>> typeReference = new TypeReference<Map<String, Object>>() {
        };
        Map<String, Object> param = JSONObject.parseObject(JSON.toJSONString(modifyUserAttributesRequest), typeReference);
        // 发送请求
        RequestMessage request = new WecloudRequestMessageBuilder().setEndpoint(MODIFY_USER_ATTRIBUTES_REQUEST)
                .setMethod(HttpMethod.POST).setParameters(param)
                .setOriginalRequest(modifyUserAttributesRequest).build();

        Object result = doOperation(request);
        Boolean flag = JSON.parseObject(JSON.toJSONString(result), Boolean.class);
        return flag;

    }
}
