package com.wecloud.im.action;

import io.geekidea.springbootplus.framework.core.pagination.Paging;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.wecloud.dispatch.annotation.ActionMapping;
import com.wecloud.dispatch.extend.ActionRequest;
import com.wecloud.im.param.GetReadersParam;
import com.wecloud.im.param.ImHistoryMessagePageParam;
import com.wecloud.im.param.MsgDeleteParam;
import com.wecloud.im.param.add.ImMsgRecall;
import com.wecloud.im.param.add.ImMsgUpdate;
import com.wecloud.im.service.ImMessageService;
import com.wecloud.im.vo.OfflineMsgDto;
import com.wecloud.im.vo.ReaderList;
import com.wecloud.im.ws.model.WsResponse;
import com.wecloud.im.ws.sender.ChannelSender;

/**
 * @Author wenzhida
 * @Date 2022/2/23 17:14
 * @Description 长连接方式处理消息
 */
@Slf4j
@Component
@ActionMapping(value = "/im/message")
@Api(value = "ws-消息API", tags = {"ws-消息"})
public class MessageAction {

    @Autowired
    private ImMessageService imMessageService;

    @Autowired
    private ChannelSender channelSender;

    /**
     * 消息撤回
     * @param request
     * @param msgRecall
     * @param reqId
     */
    @ActionMapping("/withdraw")
    @ApiOperation("消息撤回")
    public WsResponse<Map<String, Boolean>> withdraw(ActionRequest request, ImMsgRecall msgRecall, String reqId) {
        log.info("撤回消息内容：{}", JSON.toJSONString(msgRecall));
        boolean r = imMessageService.withdrawMsg(request, msgRecall);
        Map<String, Boolean> result = new HashMap<>();
        result.put("result", r);
        return WsResponse.ok(result);
    }

    /**
     * 删除消息内容
     * @param request
     * @param msgDeleteParam
     * @param reqId
     */
    @ActionMapping("/delete")
    @ApiOperation("删除消息内容")
    public WsResponse<Map<String, Boolean>> deleteMsg(ActionRequest request, MsgDeleteParam msgDeleteParam, String reqId) {
        log.info("删除消息内容：{}", JSON.toJSONString(msgDeleteParam));
        boolean r = imMessageService.deleteMsg(request, msgDeleteParam);
        Map<String, Boolean> result = new HashMap<>();
        result.put("result", r);
        return WsResponse.ok(result);
    }

    /**
     * 修改消息体
     * @param request
     * @param imMsgUpdate
     * @param reqId
     */
    @ActionMapping("/updateMsgById")
    @ApiOperation("修改消息体")
    public WsResponse<Map<String, Boolean>> updateMsgById(ActionRequest request, ImMsgUpdate imMsgUpdate, String reqId) {
        log.info("修改消息内容：{}", JSON.toJSONString(imMsgUpdate));
        boolean r = imMessageService.updateMsgById(imMsgUpdate);
        Map<String, Boolean> result = new HashMap<>();
        result.put("result", r);
        return WsResponse.ok(result);
    }

    /**
     * 查询某个会话历史消息分页列表
     * @param request
     * @param param
     * @param reqId
     * @return
     */
    @ActionMapping("/getHistoryMsg")
    @ApiOperation("查询某个会话历史消息分页列表")
    public WsResponse<Paging<OfflineMsgDto>> getHistoryMsg(ActionRequest request, ImHistoryMessagePageParam param, String reqId) {
        log.info("查询某个会话历史消息分页列表：{}", JSON.toJSONString(param));
        return WsResponse.ok(imMessageService.getHistoryMsgConversationId(param));
    }

    /**
     * 查询某个消息已读client列表和未读client
     * @param request
     * @param param
     * @param reqId
     * @return
     */
    @ActionMapping("/getReaders")
    @ApiOperation("查询某个消息已读client列表和未读client")
    public WsResponse<ReaderList> getReaders(ActionRequest request, GetReadersParam param, String reqId) {
        log.info("查询某个消息已读client列表和未读client：{}", JSON.toJSONString(param));
        return WsResponse.ok(imMessageService.getReaders(request, param));
    }

}
