package com.jumeirah.common.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.metadata.OrderItem;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.jumeirah.common.entity.Stroke;
import com.jumeirah.common.enums.StrokeStatusEnum;
import com.jumeirah.common.mapper.StrokeMapper;
import com.jumeirah.common.mq.Queue;
import com.jumeirah.common.param.McStrokePageParam;
import com.jumeirah.common.param.McStrokePaymentPageParam;
import com.jumeirah.common.param.StrokePageParam;
import com.jumeirah.common.service.StrokeService;
import com.jumeirah.common.vo.McStrokePaymentQueryVo;
import com.jumeirah.common.vo.McStrokeQueryVo;
import com.jumeirah.common.vo.StrokeDetailVo;
import com.jumeirah.common.vo.StrokeQueryVo;
import io.geekidea.springbootplus.framework.common.service.impl.BaseServiceImpl;
import io.geekidea.springbootplus.framework.core.pagination.PageInfo;
import io.geekidea.springbootplus.framework.core.pagination.Paging;
import io.geekidea.springbootplus.framework.shiro.jwt.JwtToken;
import lombok.extern.slf4j.Slf4j;
import org.apache.shiro.SecurityUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * 行程表 服务实现类
 *
 * @author wei
 * @since 2020-09-29
 */
@Slf4j
@Service
public class StrokeServiceImpl extends BaseServiceImpl<StrokeMapper, Stroke> implements StrokeService {

    @Autowired
    private Queue queue;

    @Autowired
    private StrokeMapper strokeMapper;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public boolean saveStroke(Stroke stroke) throws Exception {
        final boolean save = super.save(stroke);
        if (save) {//推送订单到客服系统
            queue.pushOrder(stroke);
        }
        return save;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean updateStroke(Stroke stroke) throws Exception {
        return super.updateById(stroke);
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean deleteStroke(Long id) throws Exception {
        JwtToken jwtToken = (JwtToken) SecurityUtils.getSubject().getPrincipal();
        Stroke byId = super.getById(id);

        // 如果已删除 直接返回true
        if (byId == null) {
            return true;
        }

        // 判断该记录是否属于此用户
        if (!byId.getUserId().equals(jwtToken.getUserId())) {
            return false;
        }

        return super.removeById(id);
    }

    @Override
    public boolean cancelStroke(Long id) throws Exception {
        JwtToken jwtToken = (JwtToken) SecurityUtils.getSubject().getPrincipal();
        Stroke byId = super.getById(id);
        // 判断该记录是否属于此用户
        if (!byId.getUserId().equals(jwtToken.getUserId())) {
            return false;
        }
        Stroke stroke = new Stroke();
        stroke.setId(id);
        stroke.setStatus(StrokeStatusEnum.CANCEL.getCode());
        return super.updateById(stroke);
    }

    @Override
    public StrokeDetailVo getStrokeById(Long id) throws Exception {
        return strokeMapper.getStrokeById(id);
    }

    @Override
    public Paging<StrokeQueryVo> getStrokePageList(StrokePageParam strokePageParam) throws Exception {
//        Page<StrokeQueryVo> page = new PageInfo<>(strokePageParam, OrderItem.asc(getLambdaColumn(Stroke::getCreateTime)));
        Page<StrokeQueryVo> page = new PageInfo<>(strokePageParam, OrderItem.desc("s.create_time"));
        JwtToken jwtToken = (JwtToken) SecurityUtils.getSubject().getPrincipal();
        // 查询已完成 需要额外查询已取消状态
        if (strokePageParam.getStatus().equals(StrokeStatusEnum.COMPLETED.getCode())) {
            IPage<StrokeQueryVo> iPage = strokeMapper.getStrokePageListWithFinish(page, strokePageParam, jwtToken.getUserId());
            return new Paging<StrokeQueryVo>(iPage);
        }
        IPage<StrokeQueryVo> iPage = strokeMapper.getStrokePageList(page, strokePageParam, jwtToken.getUserId());
        return new Paging<StrokeQueryVo>(iPage);
    }

    @Override
    public Paging<McStrokeQueryVo> getMcStrokePageList(McStrokePageParam mcStrokePageParam) {
        Page<StrokeQueryVo> page = new PageInfo<>(mcStrokePageParam, OrderItem.asc(getLambdaColumn(Stroke::getCreateTime)));
        IPage<McStrokeQueryVo> mcStrokePageList = strokeMapper.getMcStrokePageList(page, mcStrokePageParam);
        return new Paging<McStrokeQueryVo>(mcStrokePageList);
    }

    @Override
    public Paging<McStrokePaymentQueryVo> getPaymentPageList(McStrokePaymentPageParam mcStrokePaymentPageParam) {
        Page<StrokeQueryVo> page = new PageInfo<>(mcStrokePaymentPageParam);
        IPage<McStrokePaymentQueryVo> mcStrokePageList = strokeMapper.getPaymentPageList(page, mcStrokePaymentPageParam);
        return new Paging<McStrokePaymentQueryVo>(mcStrokePageList);
    }

}
