package com.wecloud.im.security.signature;

import lombok.extern.slf4j.Slf4j;

import java.io.UnsupportedEncodingException;
import java.util.Map;
import java.util.TreeMap;

import cn.hutool.core.lang.Assert;


/**
 * 签名工具类
 * @Author luozh
 * @Date 2022年04月14日 11:32o
 * @Version 1.0
 */
@Slf4j
public class SignUtils {

    public static final String IM_PREFIX = "x-im-";

    public static final String CONTENT_TYPE = "Content-Type";

    public static final String DATE = "Date";

    public static final String NEW_LINE = "\n";

    public static String buildCanonicalString(String method, String resourcePath, Map<String, String> headers) {
        StringBuilder canonicalString = new StringBuilder();
        canonicalString.append(method).append(NEW_LINE);

        TreeMap<String, String> fixedHeadersToSign = new TreeMap<>();
        TreeMap<String, String> canonicalizedImHeadersToSign = new TreeMap<>();

        if (headers != null) {
            for (Map.Entry<String, String> header : headers.entrySet()) {
                if (header.getKey() != null) {
                    String lowerKey = header.getKey().toLowerCase();
                    if (lowerKey.equals(CONTENT_TYPE.toLowerCase())) {
                        fixedHeadersToSign.put(lowerKey, header.getValue().trim());
                    } else if (lowerKey.equals(DATE.toLowerCase())) {
                        fixedHeadersToSign.put(lowerKey, header.getValue().trim());
                    } else if (lowerKey.startsWith(IM_PREFIX)) {
                        canonicalizedImHeadersToSign.put(lowerKey, header.getValue().trim());
                    }
                }
            }
        }

        if (!fixedHeadersToSign.containsKey(CONTENT_TYPE.toLowerCase())) {
            fixedHeadersToSign.put(CONTENT_TYPE.toLowerCase(), "");
        }

        // Append fixed headers to sign to canonical string
        for (Map.Entry<String, String> entry : fixedHeadersToSign.entrySet()) {
            Object value = entry.getValue();

            canonicalString.append(value);
            canonicalString.append(NEW_LINE);
        }

        // Append canonicalized im headers to sign to canonical string
        for (Map.Entry<String, String> entry : canonicalizedImHeadersToSign.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();

            canonicalString.append(key).append(':').append(value).append(NEW_LINE);
        }

        // Append canonical resource to canonical string
        canonicalString.append(buildCanonicalizedResource(resourcePath));

        return canonicalString.toString();
    }

    private static String buildCanonicalizedResource(String resourcePath) {
        Assert.isTrue(resourcePath.startsWith("/"), "Resource path should start with slash character");

        StringBuilder builder = new StringBuilder();
        builder.append(uriEncoding(resourcePath));
        return builder.toString();
    }

    public static String uriEncoding(String uri) {
        String result = "";

        try {
            for (char c : uri.toCharArray()) {
                if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z')
                        || (c >= '0' && c <= '9') || c == '_' || c == '-'
                        || c == '~' || c == '.') {
                    result += c;
                } else if (c == '/') {
                    result += "%2F";
                } else {
                    byte[] b;
                    b = Character.toString(c).getBytes("utf-8");

                    for (int i = 0; i < b.length; i++) {
                        int k = b[i];

                        if (k < 0) {
                            k += 256;
                        }
                        result += "%" + Integer.toHexString(k).toUpperCase();
                    }
                }
            }
        } catch (UnsupportedEncodingException e) {
            throw new IllegalStateException(e);
        }
        return result;
    }

    public static String buildSignature(String secretAccessKey, String httpMethod, String resourcePath, Map<String, String> headers) {
        String canonicalString = buildCanonicalString(httpMethod, resourcePath, headers);
        return new HmacSHA256Signature().computeSignature(secretAccessKey, canonicalString);
    }

    /**
     * 创建回调 Signature (数据签名)
     * @Author luozh
     * @Date 2022年04月22日 08:43:04
     * @param appKey 蔚可云平台分配的App Key
     * @param appSecret 蔚可云平台分配的App Secret
     * @param nonce 随机字符串 长度为21
     * @param date GMT格式时间8
     * @Return 数据签名
     */
    public static String buildSignature(String appKey, String appSecret, String nonce, String date) {
        String canonicalString = appKey + nonce + date;
        return new HmacSHA256Signature().computeSignature(appSecret, canonicalString);
    }

}
