package com.wecloud.im.action;

import io.geekidea.springbootplus.framework.core.pagination.Paging;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.alibaba.fastjson.JSON;
import com.wecloud.dispatch.annotation.ActionMapping;
import com.wecloud.im.param.GetReadersParam;
import com.wecloud.im.param.ImHistoryMessagePageParam;
import com.wecloud.im.param.MsgDeleteParam;
import com.wecloud.im.param.add.ImMsgRecall;
import com.wecloud.im.param.add.ImMsgUpdate;
import com.wecloud.im.service.ImMessageService;
import com.wecloud.im.vo.OfflineMsgDto;
import com.wecloud.im.vo.ReaderList;
import com.wecloud.im.ws.model.WsResponse;

/**
 * @Author wenzhida
 * @Date 2022/2/23 17:14
 * @Description 长连接方式处理消息
 */
@Slf4j
@Component
@ActionMapping(value = "/im/message")
@Api(value = "ws-消息API", tags = {"ws-消息"})
public class MessageAction {

    @Autowired
    private ImMessageService imMessageService;

    /**
     * 消息撤回
     * @param request
     * @param msgRecall
     * @param reqId
     */
    @ActionMapping("/withdraw")
    @ApiOperation("消息撤回")
    public WsResponse<Map<String, Boolean>> withdraw(ImMsgRecall data) {
        log.info("撤回消息内容：{}", JSON.toJSONString(data));
        Boolean r = imMessageService.withdrawMsg(data);
        Map<String, Boolean> result = new HashMap<>();
        result.put("result", r);
        return WsResponse.ok(result);
    }

    /**
     * 删除消息内容
     * @param request
     * @param msgDeleteParam
     * @param reqId
     */
    @ActionMapping("/delete")
    @ApiOperation("删除消息内容")
    public WsResponse<Map<String, Boolean>> deleteMsg(MsgDeleteParam data) {
        log.info("删除消息内容：{}", JSON.toJSONString(data));
        boolean r = imMessageService.deleteMsg(data);
        Map<String, Boolean> result = new HashMap<>();
        result.put("result", r);
        return WsResponse.ok(result);
    }

    /**
     * 修改消息体
     * @param request
     * @param imMsgUpdate
     * @param reqId
     */
    @ActionMapping("/updateMsgById")
    @ApiOperation("修改消息体")
    public WsResponse<Map<String, Boolean>> updateMsgById(ImMsgUpdate data) {
        log.info("修改消息内容：{}", JSON.toJSONString(data));
        boolean r = imMessageService.updateMsgById(data);
        Map<String, Boolean> result = new HashMap<>();
        result.put("result", r);
        return WsResponse.ok(result);
    }

    /**
     * 查询某个会话历史消息分页列表
     * @param request
     * @param param
     * @param reqId
     * @return
     */
    @ActionMapping("/getHistoryMsg")
    @ApiOperation("查询某个会话历史消息分页列表")
    public WsResponse<Paging<OfflineMsgDto>> getHistoryMsg(ImHistoryMessagePageParam data) {
        log.info("查询某个会话历史消息分页列表：{}", JSON.toJSONString(data));
        return WsResponse.ok(imMessageService.getHistoryMsgConversationId(data));
    }

    /**
     * 查询某个会话历史消息分页列表
     * @param request
     * @param param
     * @param reqId
     * @return
     */
    @ActionMapping("/getHistoryMsgNew")
    @ApiOperation("查询某个会话历史消息分页列表-新")
    public WsResponse<Paging<OfflineMsgDto>> getHistoryMsgNew(ImHistoryMessagePageParam data) {
        log.info("查询某个会话历史消息分页列表-新：{}", JSON.toJSONString(data));
        return WsResponse.ok(imMessageService.getHistoryMsgConversationIdNew(data));
    }

    /**
     * 查询某个消息已读client列表和未读client
     * @param request
     * @param param
     * @param reqId
     * @return
     */
    @ActionMapping("/getReaders")
    @ApiOperation("查询某个消息已读client列表和未读client")
    public WsResponse<ReaderList> getReaders(GetReadersParam data) {
        log.info("查询某个消息已读client列表和未读client：{}", JSON.toJSONString(data));
        return WsResponse.ok(imMessageService.getReaders(data));
    }

}
