package com.wecloud.im.service;

import com.baomidou.lock.annotation.Lock4j;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.wecloud.im.entity.ImClient;
import com.wecloud.im.entity.ImConversationMembers;
import com.wecloud.im.entity.ImMessage;
import com.wecloud.im.param.ImConversationQueryVo;
import com.wecloud.im.ws.utils.RedisUtils;
import com.wecloud.utils.SnowflakeUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.Date;
import java.util.List;


/**
 * @Author Future
 * @Date 2022/8/24 17:18
 * @Description 本地缓存服务
 */
@Slf4j
@Service
public class SaveMessageService {

    private static final String LAST_MSG_PREFIX = "last_msg_";


    @Autowired
    private ImMessageService imMessageService;

    @Autowired
    private RedisUtils redisUtils;

    /**
     * 持久化消息到数据库并更新相关缓存
     * expire 锁过期时间 防止死锁
     *
     * @param imMessage
     */
    @Lock4j(keys = {"#imMessage.fkConversationId"}, expire = 3000, acquireTimeout = 1000)
    public void saveMessageToDb(ImMessage imMessage) {
        imMessage.setId(SnowflakeUtil.getId());
        imMessage.setCreateTime(new Date());
        this.setLastMessageIdAndFreshCash(imMessage);
        imMessageService.save(imMessage);
    }

    /**
     * 查找会话的最新一条消息
     *
     * @param message
     * @return
     */
    private void setLastMessageIdAndFreshCash(ImMessage message) {
        try {
            // 执行业务方法
            String key = LAST_MSG_PREFIX + message.getFkConversationId();
            String messageId = redisUtils.getKey(key);
            if (StringUtils.isNotBlank(messageId)) {
                // 将Redis中最新的messageId存入
                message.setPreMessageId(Long.valueOf(messageId));
            } else {
                // 落库查
                ImMessage preMessage = imMessageService.getOne(new QueryWrapper<ImMessage>().lambda()
                        .eq(ImMessage::getFkConversationId, message.getFkConversationId())
                        .orderByDesc(ImMessage::getId)
                        .last("limit 1")
                );
                if (preMessage != null) {
                    message.setPreMessageId(preMessage.getId());
                }
            }
            redisUtils.addKey(key, message.getId().toString(), Duration.ofMinutes(30));
        } catch (Exception e) {
            log.info("获取会话最后一条消息异常 ", e);
        }
    }

}
