/*
 * Copyright 2019-2029 geekidea(https://github.com/geekidea)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.wecloud.im.core.common.api;

import com.wecloud.im.core.config.il8n.I18nMessageUtil;
import com.wecloud.im.core.config.il8n.LanguageEnum;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.experimental.Accessors;

import java.io.IOException;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>
 * REST API 返回结果
 * 国际化message封装
 * </p>
 *
 * @author geekidea
 * @since 2018-11-08
 */
@Data
@Accessors(chain = true)
@Builder
@AllArgsConstructor
public class ApiResult<T> implements Serializable {
    private static final long serialVersionUID = 8004487252556526569L;

    /**
     * 响应码
     */
    private int code;

    /**
     * 响应消息
     */
    private String message;

    /**
     * 响应数据
     */
    private T data;


    public ApiResult() {
    }

    public static ApiResult<Boolean> result(boolean flag) {
        if (flag) {
            return ok();
        }
        return fail();
    }


    public static ApiResult<Boolean> result(ApiCode apiCode) {
        return result(apiCode, null);
    }

    public static <T> ApiResult<T> result(ApiCode apiCode, T data) {
        return result(apiCode, null, data);
    }

    public static <T> ApiResult<T> result(ApiCode apiCode, T data, String language) {
        return result(apiCode, null, data);
    }


    public static <T> ApiResult<T> result(ApiCode apiCode, String message, T data) {
//        boolean success = false;
//        if (apiCode.getCode() == ApiCode.SUCCESS.getCode()) {
//            success = true;
//        }

        // 多语言国际化,根据http上下文, 取得heard中的language语言属性,实现不用在业务代码中传递语言字段
//        HttpServletRequest request = HttpServletRequestUtil.getRequest();
//        String language = request.getHeader("language");

        String success = "SUCCESS";
        try {
            message = I18nMessageUtil.getMessage(LanguageEnum.getLanguageType(null), apiCode.getMessage(), success);
        } catch (IOException e) {
            message = success;
        }


        return (ApiResult<T>) ApiResult.builder()
                .code(apiCode.getCode())
                .message(message)
                .data(data)
//                .success(success)
//                .time(new Date())
                .build();
    }

    public static ApiResult<Boolean> ok() {
        return ok(true);
    }

    public static <T> ApiResult<T> ok(T data) {
        return result(ApiCode.SUCCESS, data);
    }


//    public static <T> ApiResult<T> ok(T data, String message) {
//        return result(ApiCode.SUCCESS, message, data);
//    }

//    public static ApiResult<Map<String, Object>> okMap(String key, Object value) {
//        Map<String, Object> map = new HashMap<>(1);
//        map.put(key, value);
//        return ok(map);
//    }

    public static ApiResult<Boolean> fail(ApiCode apiCode) {
        return result(apiCode, null);
    }

    public static <T> ApiResult<T> fail(ApiCode apiCode, String language) {
        return result(apiCode, null);
    }


    public static <T> ApiResult<T> fail(ApiCode apiCode, T data) {
        if (ApiCode.SUCCESS == apiCode) {
            throw new RuntimeException("失败结果状态码不能为" + ApiCode.SUCCESS.getCode());
        }
        return result(apiCode, data);
    }

    public static ApiResult<String> fail(Integer errorCode, String message) {
        return new ApiResult<String>()
//                .setSuccess(false)
                .setCode(errorCode)
                .setMessage(message);
    }

    public static ApiResult<Map<String, Object>> fail(String key, Object value) {
        Map<String, Object> map = new HashMap<>(1);
        map.put(key, value);
        return result(ApiCode.FAIL, map);
    }

    public static ApiResult<Boolean> fail() {
        return fail(ApiCode.FAIL);
    }

    public static ApiResult<Boolean> fail(String language) {
        return fail(ApiCode.FAIL, language);
    }
}
