package com.wecloud.im.chatroom.cache;

import com.wecloud.im.server.utils.RedisUtils;
import com.wecloud.im.core.util.GetIpUtils;
import lombok.extern.slf4j.Slf4j;

import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 聊天室的缓存（本地缓存待实现）
 *
 * @author lixiaozhong
 * @date 2022年01月4日 17:00:00
 */
@Service
@Slf4j
public class ChatRoomCacheManager {

    /**
     * redis的群聊天室房间键 key
     */
    public static final String CHAT_MEMBER_KEY = "chat_member_";

    @Resource
    private RedisUtils redisUtils;

    /**
     * 成员进入聊天室
     *
     * @param clientId
     * @param chatRoomId
     * @param platform
     */
    public void intoRoom(Long clientId, Long chatRoomId, Integer platform) {
        redisUtils.hashset(CHAT_MEMBER_KEY + chatRoomId, clientId + RedisUtils.SPLIT + platform, GetIpUtils.getlanIp(),
                2, TimeUnit.DAYS);

    }

    /**
     * 成员离开聊天室
     *
     * @param clientId
     * @param chatRoomId
     * @param platform
     */
    public void exitRoom(Long clientId, Long chatRoomId, Integer platform) {
        redisUtils.hashdel(CHAT_MEMBER_KEY + chatRoomId, clientId + RedisUtils.SPLIT + platform);
    }

    /**
     * 根据群ID 获取 聊天室的 在线成员的key-val
     *
     * @param chatRoomId
     * @return 在线成员的key-val，其中key是 client的主键id:platform, val是 ip
     */
    public Map<String, String> findOnlineClientsByChatRoomId(Long chatRoomId) {
        return redisUtils.hashgetll(CHAT_MEMBER_KEY + chatRoomId);
    }

    /**
     * 根据群ID 获取 聊天室的 在线成员的ip以及对应的成员信息，key-val
     *
     * @param chatRoomId
     * @return 在线成员的key-val，其中key是 ip地址, val是 client的主键id:platform
     */
    public Map<String, List<String>> findOnlineHostsByChatRoomId(Long chatRoomId) {
        Map<String /** client的主键ID:platform **/, String /** ip **/> onlineClientIpMap = findOnlineClientsByChatRoomId(chatRoomId);
        Map<String /** ip **/, List<String /** client的主键ID:platform **/>> onlineIpClientMap = new HashMap<>();
        onlineClientIpMap.forEach((clientIdAndPlatforms, ip) -> {
            onlineIpClientMap.putIfAbsent(ip, new ArrayList<>());
            onlineIpClientMap.get(ip).add(clientIdAndPlatforms);
        });
        return onlineIpClientMap;
    }

}
